import logging
import evdev
import threading
import queue
import time

logger = logging.getLogger(__name__)

# evdev keycodes are documented in https://github.com/torvalds/linux/blob/master/include/uapi/linux/input-event-codes.h

# Mapping of evdev keycodes to ASCII, consistent with TI-99/4A mappings for control characters
KEYBOARD_KEYCODE_TO_ASCII = {
    evdev.ecodes.KEY_A: ord('a'),
    evdev.ecodes.KEY_B: ord('b'),
    evdev.ecodes.KEY_C: ord('c'),
    evdev.ecodes.KEY_D: ord('d'),
    evdev.ecodes.KEY_E: ord('e'),
    evdev.ecodes.KEY_F: ord('f'),
    evdev.ecodes.KEY_G: ord('g'),
    evdev.ecodes.KEY_H: ord('h'),
    evdev.ecodes.KEY_I: ord('i'),
    evdev.ecodes.KEY_J: ord('j'),
    evdev.ecodes.KEY_K: ord('k'),
    evdev.ecodes.KEY_L: ord('l'),
    evdev.ecodes.KEY_M: ord('m'),
    evdev.ecodes.KEY_N: ord('n'),
    evdev.ecodes.KEY_O: ord('o'),
    evdev.ecodes.KEY_P: ord('p'),
    evdev.ecodes.KEY_Q: ord('q'),
    evdev.ecodes.KEY_R: ord('r'),
    evdev.ecodes.KEY_S: ord('s'),
    evdev.ecodes.KEY_T: ord('t'),
    evdev.ecodes.KEY_U: ord('u'),
    evdev.ecodes.KEY_V: ord('v'),
    evdev.ecodes.KEY_W: ord('w'),
    evdev.ecodes.KEY_X: ord('x'),
    evdev.ecodes.KEY_Y: ord('y'),
    evdev.ecodes.KEY_Z: ord('z'),
    evdev.ecodes.KEY_1: ord('1'),
    evdev.ecodes.KEY_2: ord('2'),
    evdev.ecodes.KEY_3: ord('3'),
    evdev.ecodes.KEY_4: ord('4'),
    evdev.ecodes.KEY_5: ord('5'),
    evdev.ecodes.KEY_6: ord('6'),
    evdev.ecodes.KEY_7: ord('7'),
    evdev.ecodes.KEY_8: ord('8'),
    evdev.ecodes.KEY_9: ord('9'),
    evdev.ecodes.KEY_0: ord('0'),
    # enter key mapped per TI-99/4A ROM
    evdev.ecodes.KEY_ENTER: ord('\r'),
    evdev.ecodes.KEY_SPACE: ord(' '),
    evdev.ecodes.KEY_BACKSPACE: ord('\b'),
    evdev.ecodes.KEY_TAB: ord('\t'),
    evdev.ecodes.KEY_ESC: ord('\x1b'),
    evdev.ecodes.KEY_MINUS: ord('-'),
    evdev.ecodes.KEY_EQUAL: ord('='),
    evdev.ecodes.KEY_LEFTBRACE: ord('['),
    evdev.ecodes.KEY_RIGHTBRACE: ord(']'),
    evdev.ecodes.KEY_BACKSLASH: ord('\\'),
    evdev.ecodes.KEY_SEMICOLON: ord(';'),
    evdev.ecodes.KEY_APOSTROPHE: ord('\''),
    evdev.ecodes.KEY_GRAVE: ord('`'),
    evdev.ecodes.KEY_COMMA: ord(','),
    evdev.ecodes.KEY_DOT: ord('.'),
    evdev.ecodes.KEY_SLASH: ord('/'),
    # arrow keys mapped per TI-99/4A ROM
    evdev.ecodes.KEY_LEFT: ord('\b'),
    evdev.ecodes.KEY_RIGHT: ord('\t'),
    evdev.ecodes.KEY_DOWN: ord('\n'),
    evdev.ecodes.KEY_UP: ord('\v'),
    # Add more mappings as needed
}

# Mapping of evdev keycodes to ASCII, consistent with TI-99/4A mappings for control characters
KEYPAD_KEYCODE_TO_ASCII = {
    # keypad keys
    evdev.ecodes.KEY_KP0: ord('0'),
    evdev.ecodes.KEY_KP1: ord('1'),
    evdev.ecodes.KEY_KP2: ord('2'),
    evdev.ecodes.KEY_KP3: ord('3'),
    evdev.ecodes.KEY_KP4: ord('4'),
    evdev.ecodes.KEY_KP5: ord('5'),
    evdev.ecodes.KEY_KP6: ord('6'),
    evdev.ecodes.KEY_KP7: ord('7'),
    evdev.ecodes.KEY_KP8: ord('8'),
    evdev.ecodes.KEY_KP9: ord('9'),
    evdev.ecodes.KEY_KPDOT: ord('.'),
    evdev.ecodes.KEY_KPMINUS: ord('-'),
    evdev.ecodes.KEY_KPPLUS: ord('+'),
    evdev.ecodes.KEY_KPASTERISK: ord('*'),
    evdev.ecodes.KEY_KPSLASH: ord('/'),
    evdev.ecodes.KEY_KPENTER: ord('\r'),
}

SHIFT_KEYCODES = {evdev.ecodes.KEY_LEFTSHIFT, evdev.ecodes.KEY_RIGHTSHIFT}
CONTROL_KEYCODES = {evdev.ecodes.KEY_LEFTCTRL, evdev.ecodes.KEY_RIGHTCTRL}
CAPSLOCK_KEYCODE = evdev.ecodes.KEY_CAPSLOCK
NUMLOCK_KEYCODE = evdev.ecodes.KEY_NUMLOCK

class KeyboardPlugin(object):
    def __init__(self):
        logger.info('created plugin instance')
        self.device = None
        self.queue = queue.Queue()
        self.thread = None
        self.initialized = False
        self.grabbed = False
        self.shift_pressed = False
        self.control_pressed = False
        self.capslock_on = False
        self.numlock_on = False
        self.update_leds = True

    def grab_keyboard(self):
        devices = [evdev.InputDevice(path) for path in evdev.list_devices()]
        for device in devices:
            logger.info(f'examining device: {device.name}')
            if 'keyboard' in device.name.lower():
                self.device = device
                self.device.grab()
                self.grabbed = True
                self.update_leds = True
                logger.info(f'grabbed keyboard: {device.name}')
                break

    def set_leds(self):
         # set the caps lock led on/off
         if self.capslock_on:
            self.device.set_led(evdev.ecodes.LED_CAPSL, 1)
         else:
            self.device.set_led(evdev.ecodes.LED_CAPSL, 0)

         # set the num lock led on/off
         if self.numlock_on:
            self.device.set_led(evdev.ecodes.LED_NUML, 1)
         else:
            self.device.set_led(evdev.ecodes.LED_NUML, 0)
   
         self.update_leds = False

    def read_keyboard(self):

        logger.info('read_keyboard thread started')

        # loop forever

        while True:
            try:

                # grab the keyboard
                self.grab_keyboard()

                # if the keyboard was grabbed...
                if self.grabbed:

                    # update keyboard leds initially and after recovery (such as keyboard unplugged/plugged in)
                    if self.update_leds:
                        self.set_leds()
    
                    # loop continuously through events, unless an exception occurs, such as keyboard unplugged
                    for event in self.device.read_loop():
    
                        if event.type == evdev.ecodes.EV_KEY:
                            if event.code in SHIFT_KEYCODES:
                                self.shift_pressed = event.value != 0
                            elif event.code in CONTROL_KEYCODES:
                                self.control_pressed = event.value != 0
                            elif event.code == CAPSLOCK_KEYCODE and event.value == 1:
                                self.capslock_on = not self.capslock_on
                                self.set_leds()       
                            elif event.code == NUMLOCK_KEYCODE and event.value == 1:
                                self.numlock_on = not self.numlock_on
                                self.set_leds()
                            elif event.value == 1 or event.value == 0:  # Key down/up event
            
                                # handle keys on the primary keyboard
                                ascii_code = KEYBOARD_KEYCODE_TO_ASCII.get(event.code, None)
                                if ascii_code is not None:
                                    if self.control_pressed:
                                        ascii_code = ascii_code & 0x1F  # Control character
                                    elif self.shift_pressed or self.capslock_on:
                                        if ord('a') <= ascii_code <= ord('z'):
                                            ascii_code = ascii_code - ord('a') + ord('A')
                                        elif ascii_code in range(ord('1'), ord('9') + 1):
                                            ascii_code = ord('!@#$%^&*()'[ascii_code - ord('1')])
                                        elif ascii_code == ord('0'):
                                            ascii_code = ord(')')
                                        elif ascii_code == ord('-'):
                                            ascii_code = ord('_')
                                        elif ascii_code == ord('='):
                                            ascii_code = ord('+')
                                        elif ascii_code == ord('['):
                                            ascii_code = ord('{')
                                        elif ascii_code == ord(']'):
                                            ascii_code = ord('}')
                                        elif ascii_code == ord('\\'):
                                            ascii_code = ord('|')
                                        elif ascii_code == ord(';'):
                                            ascii_code = ord(':')
                                        elif ascii_code == ord('\''):
                                            ascii_code = ord('"')
                                        elif ascii_code == ord('`'):
                                            ascii_code = ord('~')
                                        elif ascii_code == ord(','):
                                            ascii_code = ord('<')
                                        elif ascii_code == ord('.'):
                                            ascii_code = ord('>')
                                        elif ascii_code == ord('/'):
                                            ascii_code = ord('?')
             
                                else: # handle keys on the keypad
            
                                    ascii_code = KEYPAD_KEYCODE_TO_ASCII.get(event.code, None)
                                    if ascii_code is not None:
                                       if not self.numlock_on:
                                          if ascii_code == ord('8'):
                                             ascii_code = ord('\v')
                                          elif ascii_code == ord('2'):
                                             ascii_code = ord('\n')
                                          elif ascii_code == ord('4'):
                                             ascii_code = ord('\b')
                                          elif ascii_code == ord('6'):
                                             ascii_code = ord('\t')
                                          else:
                                             # cancel the key event
                                             ascii_code = None
    
                                # queue the ascii code
                                if ascii_code is not None:                       
                                    # flag key_up by negating the ascii_code
                                    if event.value == 0:
                                        ascii_code = -ascii_code;
                                        self.queue.put(ascii_code)
                                        logger.info(f'key up: {evdev.ecodes.KEY[event.code]}, ascii code: {ascii_code}')
                                    else:
                                        self.queue.put(ascii_code)
                                        logger.info(f'key down: {evdev.ecodes.KEY[event.code]}, ascii code: {ascii_code}')
                                else:
                                    logger.warning(f'key code {event.code} not mapped to ASCII')
                else:
                   logger.warning('waiting for keyboard to become available')
                   time.sleep(1)
            except Exception as e:
                logger.error(f'keyreader exception: {e}')
                self.grabbed = False
                time.sleep(1)

    def handle(self, bytes):
        try:
            if not self.initialized:
                self.thread = threading.Thread(target=self.read_keyboard, daemon=True)
                self.thread.start()
                self.initialized = True

            if not self.queue.empty():
                ascii_code = self.queue.get()
                logger.info(f'handled keyboard plugin message, ascii code: {ascii_code}')
                return [ascii_code]
            else:
                time.sleep(0.001)
                return []

        except Exception as e:
            logger.error(f'something went wrong: {e}')
            return []

