/* adduser.c
   This program provides the ability to add a user account.
   Usage: adduser [-c long_name] [-d home_directory] [-g group_id] [-s shell] [-u user_id] username
   Options:
   -c sets the long name
   -d sets the home directory
   -g sets group id
   -p sets password to one generated by crypt
   -s sets the shell path
   -u sets the user id
   change log:
   11/25/2023 initial version
   12/10/2023 added reference to stdlib.h
   02/27/2024 added use of dylib
   03/01/2024 modified to make all calls to dylib.fputs use string constants rather than literals to prevent compiler translation
              to fwrite
   01/20/2025 updated fputs to dylib
   09/29/2025 default real name to 'none' when not specified
              added making home directory when it doesn't exist
   10/11/2025 addressed strtok issue in accts_read.c so real name can be defaulted to blank
*/

#include <stdbool.h>
#include <stdio.h>
#include <unistd.h>
#include <constants.h>
#include <string.h>
#include <accts_private.h>
#include <stdlib.h>
#include <dirent.h>
#include <sys/stat.h>
#include <dylib.h>

const char *usage_msg = "usage: adduser [-c long_name] [-d home_directory] [-g group_id] [-s shell] [-u user_id] username\n";
void print_usage () {
   dylib.fputs (usage_msg, stdout);
}

const char *errmsg1 = "username already in use\n";
const char *errmsg2 = "can't reading passwd file\n";
const char *errmsg3 = "adduser can only be run by root\n";

int main (int argc, char *argv[]) {

   int r      = UNDEFINED;                       // set to failed by default
   bool error = false;                           // set error state to false
 
   char username[10];                            // username
   char cipher[32];                              // cipher
   char long_name[32];                           // long name
   char home_dir[FILENAME_MAX];                  // home directory
   char shell[FILENAME_MAX];                     // shell
   int user_id  = UNDEFINED;                     // user id override
   int group_id = UNDEFINED;                     // group id override

   dylib.strcpy (username, "");                  // default the user name
   dylib.strcpy (cipher, "0F1E2D3C4B5A6978");    // set the cipher to something essentially random
   dylib.strcpy (long_name, "");                 // default the long name
   dylib.strcpy (home_dir, "");                  // default the home directory override
   dylib.strcpy (shell, "");                     // default the shell override

   // capture the options
   int opt;
   while ((opt = getopt (argc, argv, "c:d:g:p:s:u:")) != -1) { 
      switch (opt) {                            
         case 'c':                               // get a long name
            dylib.strcpy (long_name, optarg);
            break;
         case 'd':                               // get a home directory override
            dylib.strcpy (home_dir, optarg);
            break;
         case 'g':                               // get a group id override
            group_id = atoi (optarg);
            break;
         case 'p':                               // get a cipher override
            dylib.strcpy (cipher, optarg);
            break;
         case 's':                               // get a shell override
            dylib.strcpy (shell, optarg);
            break;
         case 'u':                               // get a user id override
            user_id = atoi (optarg);
            break;
         case '?':                               // process all else as errors
         default:
            error = true;
            break;
      }
   }
 
   // adjust argc and argv removing the processed options
   argc -= optind;
   argv += optind;

   if (argc == 1) {                              // capture the username
      dylib.strcpy (username, argv[0]);
   } else {
      error = true;                              // none provided so there's a syntax error
   }

   if (error) {                                  // if there's an error print usage
      print_usage ();
   } else {                                      // syntax is good

      if (getuid () == 0) {                      // ensure the user is root

         accts_t accts;                          // accts info
         int ra;

         accts_init (&accts);
         ra = accts_read (&accts);               // read the accounts info
         if (!ra) {
            // add the user
            ra = accts_add_user (&accts, username, cipher, long_name);
            if (!ra) {
               // update the user account for the override values

               if (user_id >= 0) {
                  accts_set_value (&accts, username, PASSWD_UID, NULL, user_id);
               }
               if (group_id >= 0) {
                  accts_set_value (&accts, username, PASSWD_GID, NULL, group_id);
               }
               if (dylib.strlen (home_dir)) {
                  accts_set_value (&accts, username, PASSWD_HOMEDIR, home_dir, 0);
               }
               if (dylib.strlen (shell)) {
                  accts_set_value (&accts, username, PASSWD_SHELL, shell, 0);
               }

               // write the accounts info back to persistent storage
               accts_write (&accts);

               // get the home directory, since it may have been generated by default
               accts_get_value (&accts, username, PASSWD_HOMEDIR, home_dir, NULL);

               // determine if the home directory exists
               DIR *d = opendir (home_dir);
               if (d) {        
                  closedir (d);   // directory exists; close it
               } else {
                  // make the home directory since it doesn't exist
                  int m = mkdir (home_dir, 0);
                  if (m) {  // output an error if mkdir failed
                     dylib.fputs ("cannot make directory: ", stderr);
                     dylib.fputs (home_dir, stderr);
                     dylib.fputs ("\n", stderr);
                  }
               }

               // set the return result to success
               r = 0;
            } else {
               dylib.fputs (errmsg1, stderr);
            }

            // free the accounts info
            accts_free (&accts);
         } else {
            dylib.fputs (errmsg2, stderr);
         }
      } else {
         dylib.fputs (errmsg3, stderr);
      }
   }

   // return the result
   return r; 
}
