/* bc.c
   This program provides a unix99-specific BASIC program to C conversion.

   Limitations
   - The conversion process is limited to line by line semantic analysis, thus there is no relationship maintained between converted lines
     beyond the small amount required for indentation and variable scope.
   - Variable scope handling is either global or method local. The conversion does handle block level variable scope, as provided by the C
     language. Current support provides just enough to handle type propogation and determination.
   - All statements must fit on a single line--no continuation is provided. 
   - Compound statements are not supported, although compound evaluations can be used in the print statement
   - If/then/else supports either use of line numbers on a single line, or block style on separate lines
   - Generally older BASIC constructs are not included, such as GOSUB, ON GOSUB, etc., as the newer semantic constructs are superior
   - Error handling, as provided by ON ERROR is not supported. C style if checks are the way.
   - Line numbers are supported, primarily for use by the legacy IF and GOTO syntax. Line numbers must be unique, although their numeric order
   - in the program is entirely irrelevant.
   - This version of BASIC is entirely subprogram/function-oriented. Statements other than REM/! and DIM for global variables are not supported.
   - As a conversion, the handling of strings is via typedef structs of fixed length. Strings do not have comparison or concatenation operators.
   - General syntax checking is lacking, although usually has enough of a hint at what went wrong. Needs major improvement.

   Change history:
   11/01/2025 initial version
   11/08/2025 Added a large number of BASIC standard functions
              Added case-insitivity to code
              Corrected for loop logic for explicit step expressions
              Updated built-in method generation to use the defined names to ensure name changes are consistent
              Added if blocks
              Added do while/until loops
              Added select case
   11/10/2025 Added subend keyword to mimic TI-XB yet continue to support QB's structured keywords
              Added char and sprite built-ins
              Added overrides area for translating conflicting language types and methods. Initially this handles char conversion to CHAR for the
              TI BASIC CALL CHAR method.
              Changed the vars package to entities since it already includes method return types, and added metadata for array support
   11/11/2025 adjusted entities metadata definitions
              added collection of metadata, particularly array info, when handling the dim statement
              added array translation to eval_write
   11/12/2025 added use of eval_write to assignment handling to address both the assignment and evaluation sides, since the assignment may have
              expressions as well
              added string compare method
   11/13/2025 removed use of comp_write and merged comparison operation translation into eval_write
   11/15/2025 added extensive logic for printing spaces after tokens to provoke a preferred C style "guide" format (not technically needed but
              may help anyone perusing the generated code -- certainly reads better than highly spaced or unspaced C.
   11/16/2025 added inkey
              added auto-reconfig of display at program exit
   11/22/2025 moved all basic methods and type definitions to the common library and named basic*
   11/27/2025 added display command with optional at clause
   11/28/2025 added file open/close and file handling initialization
   11/30/2025 added file print
   12/01/2025 added file eof
              added file reading
   12/03/2025 added delete
   12/05/2025 added vali, vall, vald
              added stri, strl, strd
              added gchar
              added missing long printing
              added position
   12/07/2025 added joyst
*/

/* TO DO:
   - Add syntax checking
   - String comparison functions
   - Ensure the all keywords are added to the is_keyword test method
   - Write program to communicate with TIPI-based software and get results
   - Write tipi software to launch compiler and return results
   - Stop using fprintf -- resources are excessive, especially when not used
*/

#include <stdbool.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <math.h>

// #define DEBUG

#define FILENAME_LEN 128

#define LINES_MAX 4096
#define LINE_LEN 128
#define NAME_LEN 32

#define VARTYPE_STRING 1
#define VARTYPE_INT    2
#define VARTYPE_LONG   3
#define VARTYPE_FLOAT  4
#define VARTYPE_DOUBLE 5

#define TOKEN_LT        "<"
#define TOKEN_LTE       "<="
#define TOKEN_GT        ">"
#define TOKEN_GTE       ">="
#define TOKEN_EQUAL     "="
#define TOKEN_NOT_EQUAL "<>"
#define TOKEN_NOT       "not"
#define TOKEN_AND       "and"
#define TOKEN_OR        "or"

#define BMAIN_METHOD    "bmain"

#define LINE_LABEL_HEAD "line_"

// SUPPORTED:

#define KEYWORD_CALL         "call"       
#define KEYWORD_SUB          "sub"        
#define KEYWORD_SUBEND       "subend"        
#define KEYWORD_FUNC         "func"       
#define KEYWORD_RETURN       "return"     
#define KEYWORD_REM          "rem"        
#define KEYWORD_REM_ALT      "!"          
#define KEYWORD_PRINT        "print"      
#define KEYWORD_FOR          "for"        
#define KEYWORD_TO           "to"         
#define KEYWORD_STEP         "step"       
#define KEYWORD_NEXT         "next"       
#define KEYWORD_IF           "if"         
#define KEYWORD_THEN         "then"       
#define KEYWORD_ELSE         "else"       
#define KEYWORD_GOTO         "goto"      
#define KEYWORD_END          "end"        
#define KEYWORD_DIM          "dim"        
#define KEYWORD_AS           "as"         
#define KEYWORD_INTEGER      "integer"    
#define KEYWORD_LONG         "long"       
#define KEYWORD_FLOAT        "float"      
#define KEYWORD_DOUBLE       "double"     
#define KEYWORD_STRING       "string"     
#define KEYWORD_LEFT         "left"       
#define KEYWORD_RIGHT        "right"      
#define KEYWORD_POS          "pos"        
#define KEYWORD_CONCAT       "concat"     
#define KEYWORD_COMPARE      "compare"     
#define KEYWORD_LEN          "len"        
#define KEYWORD_SEG          "seg"        
#define KEYWORD_RPT          "rpt"        
#define KEYWORD_ASC          "asc"        
#define KEYWORD_CHR          "chr"        
#define KEYWORD_VERSION      "version"    
#define KEYWORD_RANDOMIZE    "randomize"    
#define KEYWORD_SPRITE       "sprite"    
#define KEYWORD_LOCATE       "locate"    
#define KEYWORD_DISPLAY      "display"
#define KEYWORD_AT           "at"
#define KEYWORD_DISPLAY_MODE "display_mode"
#define KEYWORD_COLOR        "color"
#define KEYWORD_CHAR         "char"    
#define KEYWORD_INKEY        "inkey"
#define KEYWORD_RND          "rnd"
#define KEYWORD_RNDI         "rndi"
#define KEYWORD_PI           "pi"         
#define KEYWORD_CLEAR        "clear"      
#define KEYWORD_HCHAR        "hchar"      
#define KEYWORD_VCHAR        "vchar"      
#define KEYWORD_DO           "do"
#define KEYWORD_LOOP         "loop"
#define KEYWORD_WHILE        "while"
#define KEYWORD_UNTIL        "until"
#define KEYWORD_SELECT       "select"
#define KEYWORD_CASE         "case"
#define KEYWORD_OPEN         "open"
#define KEYWORD_OUTPUT       "output"
#define KEYWORD_APPEND       "append"
#define KEYWORD_CLOSE        "close"
#define KEYWORD_INPUT        "input"
#define KEYWORD_EOF          "eof"
#define KEYWORD_DELETE       "delete"
#define KEYWORD_VALI         "vali"
#define KEYWORD_VALL         "vall"
#define KEYWORD_VALD         "vald"
#define KEYWORD_STRI         "stri"
#define KEYWORD_STRL         "strl"
#define KEYWORD_STRD         "strd"
#define KEYWORD_GCHAR        "gchar"
#define KEYWORD_POSITION     "position"
#define KEYWORD_SCREEN       "screen"
#define KEYWORD_FONTLOAD     "fontload"
#define KEYWORD_DISPLAY_MODE_GET "display_mode_get"
#define KEYWORD_DISPLAY_ROWS_AND_COLS "display_rows_and_cols"
#define KEYWORD_CHARSET      "charset"
#define KEYWORD_JOYST        "joyst"
#define KEYWORD_COINC        "coinc"

/*
// TO BE SUPPORTED

ACCEPT - complex screen input
CHARPAT - gets the pattern definition for a char
COINC - determine if two sprites coincide, or all
SOUND - probably don't want to support as defined but do need a sound capability

DISTANCE - returns the distance between two sprites or a sprite and a single row

ABS - absolute value
COS - cosine function
EXP - returns the exponential value
LOG - returns the natural logarithm
ATN - arctangent
TAN - tangent
INT - truncates a value to the integer int ()
SGN - returns 1 if the value is positive, 0 if not
SIN - returns the sine of an angle
MAX - returns the largest of two numbers
MIN - returns the smallest of two numbers
SQR - square root

DISPLAY…USING - complex screen printing capability
IMAGE - format for the complex printing
PRINT USING - complex formatting
TAB - move to the next tab printing position

BYE  - returns to the TI boot screen

DELSPRITE - deletes one or more sprites or all
MAGNIFY - sets sprite magnification
MOTION - sets sprite motion
PATTERN - sets a sprite charter/s pattern

SAY - speaks with the speech synthesizer
SPGET - returns the speech data for a given string

// NOT SUPPORTED:

DEF - defines a function by expression -- no need with FUNC support
BREAK - used for debugging
CONTINUE - used for debugging
DATA - in program data, likely will not support
ERR - returns the error code of something that went wrong
GOSUB - calls a subprogram
INIT - no support
LET - not supported
LINK - not supported
LIST - not supported
LOAD - not supported
MERGE - not supported
NEW - not supported
NUMBER - not supported
ON BREAK - not supported
ON ERROR - not supported
ON GOTO - not supported, use select case
ON GOSUB - not supported
ON WARNING - not supported
OLD - not supported
OPTION BASE - not supported
PEEK - not supported
READ - not supported
REC - not supported
RETURN (with ON GOSUB) - not supported
RETURN (with ON ERROR) - not supported
RESEQUENCE - not supported
RESTORE - not supported
RUN - not supported
SAVE - not supported
SIZE - not supported
STOP - not supported
SUBEXIT - not supported - use RETURN
TRACE - not supported
UNBREAK - not supported

*/

#define max(a,b) \
   ({ __typeof__ (a) _a = (a); \
       __typeof__ (b) _b = (b); \
     _a > _b ? _a : _b; })

typedef struct {
   char name[NAME_LEN];
   int var_type;
   bool is_global;
   bool is_array;
   int array_indices_count;
} entity_t;

typedef struct {
   entity_t entry[256];
   int count;
   bool in_method;
} entities_t;

entities_t entities;

int format_indent_level;

void format_init () {
   format_indent_level = 0;
}

void format_indent () {
   format_indent_level++;
}

void format_outdent () {
   format_indent_level--;
}

void format_print () {
   for (int i = 0; i < format_indent_level; i++) {
      fprintf (stdout, "   ");
   }
}

int select_case_level = -1;
bool select_case_is_first_case[16];

void string_convert_non_literals_to_lower_case (char *s) {
   bool is_in_string_literal = false;
   for (int i = 0; i < strlen (s); i++) {
      if (s[i] == '"') {
         is_in_string_literal = !is_in_string_literal;
      }
      if (!is_in_string_literal) {
         s[i] = tolower (s[i]);
      }
   }
}

void entities_init () {
   entities.count     = 0;
   entities.in_method = false;
}

void entities_set_method_entry () {
   entities.in_method = true;
}

void entities_print () {
   #ifdef DEBUG
      fprintf 
         (stdout, 
          "// entities:\n// %-32s %6s %4s %5s %5s\n", 
          "name", 
          "global", 
          "type", 
          "array", 
          "width");
      for (int i = 0; i < entities.count; i++) {
         fprintf 
            (stdout, 
             "// %-32s %6d %4d %5d %5d\n", 
             entities.entry[i].name, 
             entities.entry[i].is_global, 
             entities.entry[i].var_type,
             entities.entry[i].is_array,
             entities.entry[i].array_indices_count);
      }
   #endif
}

void entities_clear_local () {
   for (int i = 0; i < entities.count; i++) {
      if (!entities.entry[i].is_global) {
         entities.count = i;
         break;
      }
   }
}

void entities_set_method_exit () {
   entities.in_method = false;
   entities_clear_local ();
}

void entities_add (const char *name, int var_type, bool is_array, int array_indices_count) {
   entities.entry[entities.count].var_type            = var_type;
   entities.entry[entities.count].is_global           = !entities.in_method;
   entities.entry[entities.count].is_array            = is_array;
   entities.entry[entities.count].array_indices_count = array_indices_count;
   strcpy (entities.entry[entities.count].name, name);
   entities.count++;
}

int entities_get_var_type (const char *name) {
   int r = 0;
   for (int i = 0; i < entities.count; i++) {
      if (!strcmp (name, entities.entry[i].name)) {
         r = entities.entry[i].var_type;
         break;
      }
   }
   return r;
}

bool entities_is_array (const char *name) {
   bool r = false; 
   for (int i = 0; i < entities.count; i++) {
      if (!strcmp (name, entities.entry[i].name)) {
         r = entities.entry[i].is_array;
         break;
      }
   }  
   return r;
}  

void write_includes () {
   fprintf (stdout, "#include <stdio.h>\n");
   fprintf (stdout, "#include <stdlib.h>\n");
   fprintf (stdout, "#include <basic.h>\n");
   fprintf (stdout, "#include <console.h>\n");
   fprintf (stdout, "\n");
}

void write_built_in_basic_methods () {

   entities_add (KEYWORD_CONCAT, VARTYPE_STRING, false, 0);
   entities_add (KEYWORD_COMPARE, VARTYPE_INT, false, 0);
   entities_add (KEYWORD_LEFT, VARTYPE_STRING, false, 0);
   entities_add (KEYWORD_LEN, VARTYPE_INT, false, 0);
   entities_add (KEYWORD_SEG, VARTYPE_STRING, false, 0);
   entities_add (KEYWORD_RIGHT, VARTYPE_STRING, false, 0);
   entities_add (KEYWORD_POS, VARTYPE_INT, false, 0);
   entities_add (KEYWORD_ASC, VARTYPE_INT, false, 0);
   entities_add (KEYWORD_CHR, VARTYPE_STRING, false, 0);
   entities_add (KEYWORD_VERSION, VARTYPE_DOUBLE, false, 0);
   entities_add (KEYWORD_PI, VARTYPE_DOUBLE, false, 0);
   entities_add (KEYWORD_RPT, VARTYPE_STRING, false, 0);
   entities_add (KEYWORD_RND, VARTYPE_DOUBLE, false, 0);
   entities_add (KEYWORD_RNDI, VARTYPE_INT, false, 0);
   entities_add (KEYWORD_INKEY, VARTYPE_INT, false, 0);
   entities_add (KEYWORD_AT, VARTYPE_INT, false, 0);
   entities_add (KEYWORD_EOF, VARTYPE_INT, false, 0);
   entities_add (KEYWORD_VALL, VARTYPE_LONG, false, 0);
   entities_add (KEYWORD_VALD, VARTYPE_DOUBLE, false, 0);
   entities_add (KEYWORD_VALI, VARTYPE_INT, false, 0);
   entities_add (KEYWORD_STRL, VARTYPE_STRING, false, 0);
   entities_add (KEYWORD_STRD, VARTYPE_STRING, false, 0);
   entities_add (KEYWORD_STRI, VARTYPE_STRING, false, 0);
}

bool is_keyword (const char *token) {
   bool r = false;
   if (!strcmp (token, KEYWORD_CALL)      ||
       !strcmp (token, KEYWORD_SUB)       ||
       !strcmp (token, KEYWORD_SUBEND)    ||
       !strcmp (token, KEYWORD_FUNC)      ||
       !strcmp (token, KEYWORD_RETURN)    ||
       !strcmp (token, KEYWORD_REM)       ||
       !strcmp (token, KEYWORD_REM_ALT)   ||
       !strcmp (token, KEYWORD_PRINT)     ||
       !strcmp (token, KEYWORD_FOR)       ||
       !strcmp (token, KEYWORD_TO)        ||
       !strcmp (token, KEYWORD_STEP)      ||
       !strcmp (token, KEYWORD_NEXT)      ||
       !strcmp (token, KEYWORD_IF)        ||
       !strcmp (token, KEYWORD_THEN)      ||
       !strcmp (token, KEYWORD_ELSE)      ||
       !strcmp (token, KEYWORD_GOTO)      ||
       !strcmp (token, KEYWORD_END)       ||
       !strcmp (token, KEYWORD_DIM)       ||
       !strcmp (token, KEYWORD_AS)        ||
       !strcmp (token, KEYWORD_LEFT)      ||
       !strcmp (token, KEYWORD_RIGHT)     ||
       !strcmp (token, KEYWORD_POS)       ||
       !strcmp (token, KEYWORD_CONCAT)    ||
       !strcmp (token, KEYWORD_COMPARE)   ||
       !strcmp (token, KEYWORD_LEN)       ||
       !strcmp (token, KEYWORD_SEG)       ||
       !strcmp (token, KEYWORD_RPT)       ||
       !strcmp (token, KEYWORD_ASC)       ||
       !strcmp (token, KEYWORD_CHR)       ||
       !strcmp (token, KEYWORD_PI)        ||
       !strcmp (token, KEYWORD_RANDOMIZE) ||
       !strcmp (token, KEYWORD_RND)       ||
       !strcmp (token, KEYWORD_RNDI)      ||
       !strcmp (token, KEYWORD_VERSION)   ||
       !strcmp (token, KEYWORD_CLEAR)     ||
       !strcmp (token, KEYWORD_DISPLAY_MODE) ||
       !strcmp (token, KEYWORD_DISPLAY)   ||
       !strcmp (token, KEYWORD_AT)        ||
       !strcmp (token, KEYWORD_COLOR)     ||
       !strcmp (token, KEYWORD_INKEY)     ||
       !strcmp (token, KEYWORD_DO)        ||
       !strcmp (token, KEYWORD_LOOP)      ||
       !strcmp (token, KEYWORD_WHILE)     ||
       !strcmp (token, KEYWORD_UNTIL)     ||
       !strcmp (token, KEYWORD_SELECT)    ||
       !strcmp (token, KEYWORD_CASE)      ||
       !strcmp (token, KEYWORD_OPEN)      ||
       !strcmp (token, KEYWORD_CLOSE)     ||
       !strcmp (token, KEYWORD_DELETE)    ||
       !strcmp (token, KEYWORD_INPUT)     ||
       !strcmp (token, KEYWORD_INTEGER)   ||
       !strcmp (token, KEYWORD_LONG)      ||
       !strcmp (token, KEYWORD_FLOAT)     ||
       !strcmp (token, KEYWORD_DOUBLE)    ||
       !strcmp (token, KEYWORD_STRING)) {

      r = true;
   }
   return r;
}

bool is_whitespace (char c) {
   return c <= ' ' && c != 0x00;
}

bool is_alpha (char c) {
   return (c >= 'a' && c <= 'z') || 
          (c >= 'A' && c <= 'Z');
}

bool is_digit (char c) {
   return c >= '0' && c <= '9';
}

bool is_float_digit (char c) {
   return is_digit (c) || c == '.' || c == 'e';
}

bool is_alpha_digit (char c) {
   return (c >= 'a' && c <= 'z') || 
          (c >= 'A' && c <= 'Z') || 
          (c >= '0' && c <= '9');
}

bool is_alpha_digit_underscore (char c) {
   return is_alpha_digit (c) || c == '_';
}

#define TOKEN_OTHERS "+*^/%;(),&!#:"

char *get_token (char *s, char *token) {

   char *start;

   while (is_whitespace (*s)) {                                   // skip whitespace
      s++;
   }

   if (!*s) {                                                     // handle end of line
      strcpy (token, "");
      return NULL;

      // these next set of tests are order sensitive. For instance <> and <= must come before < due to the use of strncmp()

   } else if (!strncmp (s, TOKEN_NOT_EQUAL, strlen (TOKEN_NOT_EQUAL))) {      // handle not equal
      strcpy (token, TOKEN_NOT_EQUAL);
      s += strlen (TOKEN_NOT_EQUAL);
   } else if (!strncmp (s, TOKEN_LTE, strlen (TOKEN_LTE))) {      // handle less than or equal
      strcpy (token, TOKEN_LTE);
      s += strlen (TOKEN_LTE);
   } else if (!strncmp (s, TOKEN_LT, strlen (TOKEN_LT))) {        // handle less than
      strcpy (token, TOKEN_LT);
      s += strlen (TOKEN_LTE);
   } else if (!strncmp (s, TOKEN_GTE, strlen (TOKEN_GTE))) {      // handle greater than or equal
      strcpy (token, TOKEN_GTE);
      s += strlen (TOKEN_GTE);
   } else if (!strncmp (s, TOKEN_GT, strlen (TOKEN_GT))) {        // handle greater than
      strcpy (token, TOKEN_GT);
      s += strlen (TOKEN_GT);
   } else if (!strncmp (s, TOKEN_EQUAL, strlen (TOKEN_EQUAL))) {  // handle equal and assignment
      strcpy (token, TOKEN_EQUAL);
      s += strlen (TOKEN_EQUAL);

   } else if (strchr (TOKEN_OTHERS, *s)) {                        // handle other misc other tokens
      *token = *s;
      token++;
      *token = 0x00;
      s++;
   } else if (*s == '"') {                                        // handle quoted string
      start = s;
      s++;
      while (*s != '"' && *s != 0x00) {
         s++;
      }
      s++;                                                       
      strncpy (token, start, s - start);
      token[s - start] = 0x00;
   } else if (*s == '-' && is_digit (*(s + 1))) {                 // handle negative numbers
      start = s;
      s++;         // skip over the negative
      while (is_float_digit (*s)) {
         s++;
      }
      strncpy (token, start, s - start);
      token[s - start] = 0x00;
   } else if (*s == '-') {                                        // handle subtraction operator
      *token = *s;
      token++;
      *token = 0x00;
      s++;
   } else if (is_digit (*s)) {                                    // handle remaining numbers. To do: need to handle floating point and exponents
      start = s;
      while (is_float_digit (*s)) {
         s++;
      }
      strncpy (token, start, s - start);
      token[s - start] = 0x00;
   } else if (is_alpha (*s)) {                                    // handle keywords, named subroutines, named functions  and variables
      start = s;
      while (is_alpha_digit_underscore (*s)) {
         s++;
      }
      strncpy (token, start, s - start);
      token[s - start] = 0x00;
   } else {                                                       // error
      strcpy (token, "ERROR");
      s++;
   }

   return s;                                                      // return the remaining pointer
}

typedef struct {
   char word[64][LINE_LEN];
   int count;
} tokens_t;

void to_uppercase (char *str) {
    while (*str) {
        *str = toupper ((unsigned char)*str);
        str++;
    }
}

void create_token_list (tokens_t *tokens, char *s, char *token) {

   int i;

   for (i = 0; i < 16; i++) {
      strcpy (tokens->word[i], "");
   }
   tokens->count = 0;

   while (s) {
      strcpy (tokens->word[tokens->count], token);
      tokens->count++;
      s = get_token (s, token);
   }

   for (i = 0; i < tokens->count; i++) {                         // handle keyword conflicts with C methods or types
      if (!strcasecmp (tokens->word[i], KEYWORD_CHAR)) {
         to_uppercase (tokens->word[i]);
      }
   }

   #ifdef DEBUG
      fprintf (stdout, "// tokens:\n");
      for (int i = 0; i < tokens->count; i++) {
          fprintf (stdout, "// %d: %s\n", i, tokens->word[i]);
      }
   #endif
}

int get_index (tokens_t *tokens, const char *token, int index) {
   int r = 0;
   for (int i = index; i < tokens->count; i++) {
      if (!strcmp (token, tokens->word[i])) {
         r = i;
         break;
      }
   }
   return r;
}

int get_type_index (tokens_t *tokens, int index) {
   int r = 0;
   int i = get_index (tokens, KEYWORD_AS, index);
   if (i) {
      i++;
      if (!strcmp (tokens->word[i], KEYWORD_STRING) ||
          !strcmp (tokens->word[i], KEYWORD_INTEGER) ||
          !strcmp (tokens->word[i], KEYWORD_LONG)    ||
          !strcmp (tokens->word[i], KEYWORD_FLOAT)   ||
          !strcmp (tokens->word[i], KEYWORD_DOUBLE)) {
         r = i;
      }
   }
   return r;
}

int get_type (const char *token) {
   int r = 0;
   if (!strcmp (token, KEYWORD_STRING)) {
      r = VARTYPE_STRING;
   } else if (!strcmp (token, KEYWORD_INTEGER)) {
      r = VARTYPE_INT;
   } else if (!strcmp (token, KEYWORD_LONG)) {
      r = VARTYPE_LONG;
   } else if (!strcmp (token, KEYWORD_FLOAT)) {
      r = VARTYPE_FLOAT;
   } else if (!strcmp (token, KEYWORD_DOUBLE)) {
      r = VARTYPE_DOUBLE;
   } else {
      fprintf (stdout, "error: invalid type\n");
      exit (0);
   }
   return r;
}

void write_type (const char *s) {
   if (!strcmp (s, KEYWORD_STRING)) {
      fprintf (stdout, "string_t", stdout);
   } else if (!strcmp (s, KEYWORD_INTEGER)) {
      fprintf (stdout, "int", stdout);
   } else if (!strcmp (s, KEYWORD_LONG)) {
      fprintf (stdout, "long", stdout);
   } else if (!strcmp (s, KEYWORD_FLOAT)) {
      fprintf (stdout, "float", stdout);
   } else if (!strcmp (s, KEYWORD_DOUBLE)) {
      fprintf (stdout, "double", stdout);
   } else {
      fprintf (stdout, "error: invalid type\n");
   }
}

void comp_write (tokens_t *tokens, int start, int end) {
   int compsym_index;
   compsym_index = get_index (tokens, TOKEN_EQUAL, start);
   if (!compsym_index) compsym_index = get_index (tokens, TOKEN_NOT_EQUAL, start);
   if (!compsym_index) compsym_index = get_index (tokens, TOKEN_NOT, start);
   if (!compsym_index) compsym_index = get_index (tokens, TOKEN_AND, start);
   if (!compsym_index) compsym_index = get_index (tokens, TOKEN_OR, start);
   if (!compsym_index) compsym_index = get_index (tokens, TOKEN_LT, start);
   if (!compsym_index) compsym_index = get_index (tokens, TOKEN_LTE, start);
   if (!compsym_index) compsym_index = get_index (tokens, TOKEN_GT, start);
   if (!compsym_index) compsym_index = get_index (tokens, TOKEN_GTE, start);
   if (!compsym_index) {
      fprintf (stdout, "no comparision operator found\n");
      exit (0);
   }
   fprintf (stdout, "// comp val 1 is %d to %d\n", start, compsym_index - 1);
   fprintf (stdout, "// comp val 2 is %d to %d\n", compsym_index + 1, end);

   for (int i = start; i <= end; i++) {
      if (!strcmp (tokens->word[i], TOKEN_EQUAL)) {
         fprintf (stdout, "==");
      } else if (!strcmp (tokens->word[i], TOKEN_NOT_EQUAL)) {
         fprintf (stdout, "!=");
      } else if (!strcmp (tokens->word[i], TOKEN_NOT)) {
         fprintf (stdout, "!");
      } else if (!strcmp (tokens->word[i], TOKEN_AND)) {
         fprintf (stdout, "&&");
      } else if (!strcmp (tokens->word[i], TOKEN_OR)) {
         fprintf (stdout, "||");
      } else {
         fprintf (stdout, "%s", tokens->word[i]);
      }
      if (i != end) {
         fprintf (stdout, " ");
      }
   }
}

int eval_get_type (tokens_t *tokens, int start, int end) {
   int r = 0;
   int k;

   for (int i = start; i <= end; i++) {
      k = entities_get_var_type (tokens->word[i]);             // process as if a variable
      r = max (r, k);                                      // process as if a variable
      if (!k) {
         if (is_digit (tokens->word[i][0])) {                 // determine if this is a number
            if (strstr (tokens->word[i], ".") ||             // check for special characters pushing this to a double
                strstr (tokens->word[i], "e")) {
               r = max (r, VARTYPE_DOUBLE);
            } else {
               r = max (r, VARTYPE_INT);                     // default to integer. TBD: perhaps should look at the value itself before assuming integer
            }
         } else if (tokens->word[i][0] == '"') {
            r = max (r, VARTYPE_STRING);
         }
      }
   }
   return r;
}

void eval_write (tokens_t *tokens, int start, int end, bool is_assignment) {

   int paren_index   = -1;
   bool in_array[32] = {false};
   int eval_type;
   bool last_token_is_array = false;
   bool token_is_array;

   bool write_space;

   for (int i = start; i <= end; i++) {

      token_is_array = entities_is_array (tokens->word[i]);
      eval_type      = eval_get_type (tokens, i, i);

      if (eval_type == VARTYPE_STRING && tokens->word[i][0] == '"') {
         fprintf (stdout, "string_assign (%s)", tokens->word[i]);
         write_space = false;
      } else {
         if (!strcmp (tokens->word[i], "(")) {

            paren_index++;
            in_array[paren_index] = last_token_is_array;

            if (in_array[paren_index]) {
               fprintf (stdout, "[");
            } else {
               fprintf (stdout, "(");
            }

            write_space = false;

         } else if (!strcmp (tokens->word[i], ",")) {
            if (in_array[paren_index]) {
               fprintf (stdout, "][");
               write_space = false;
            } else {
               fprintf (stdout, ",");
               write_space = true;
            }
         } else if (!strcmp (tokens->word[i], ")")) {
            if (in_array[paren_index]) {
               fprintf (stdout, "]");
            } else {
               fprintf (stdout, ")");
            }
            write_space = strcmp (tokens->word[i + 1], ")");

            paren_index--;
         } else if (!strcmp (tokens->word[i], TOKEN_EQUAL)) {
            if (is_assignment) {
               fprintf (stdout, "=");
            } else {
               fprintf (stdout, "==");
            }
         } else if (!strcmp (tokens->word[i], TOKEN_NOT_EQUAL)) {
            fprintf (stdout, "!=");
         } else if (!strcmp (tokens->word[i], TOKEN_NOT)) {
            fprintf (stdout, "!");
         } else if (!strcmp (tokens->word[i], TOKEN_AND)) {
            fprintf (stdout, "&&");
         } else if (!strcmp (tokens->word[i], TOKEN_OR)) {
            fprintf (stdout, "||");
         } else {
            fprintf (stdout, "%s", tokens->word[i]);
            if (token_is_array) {
               write_space = false;
            } else {
               write_space = !(!strcmp (tokens->word[i + 1], ")") || !strcmp (tokens->word[i + 1], ","));
            }
         }
      }

      if (write_space && i != end) {
         fprintf (stdout, " ");
      }

      last_token_is_array = token_is_array;   // save for the next token processing
   }
}

void handle_keyword_select (char *s, char *token) {
   tokens_t tokens;
   create_token_list (&tokens, s, token);
   
   format_print ();

   fprintf (stdout, "switch (");
   eval_write (&tokens, 2, tokens.count - 1, false);
   fprintf (stdout, ") {\n");
   
   format_indent ();

   select_case_level++;
   select_case_is_first_case[select_case_level] = true;
}

void handle_keyword_case (char *s, char *token) {
   tokens_t tokens; 
   create_token_list (&tokens, s, token);
   
   if (select_case_is_first_case[select_case_level]) {
      select_case_is_first_case[select_case_level] = false;
   } else {
      format_print ();
      fprintf (stdout, "break;\n");
      format_outdent ();
   }

   format_print ();

   if (!strcmp (KEYWORD_ELSE, tokens.word[1])) {
      fprintf (stdout, "default:\n");
   } else {
      fprintf (stdout, "case ");
      eval_write (&tokens, 1, tokens.count - 1, false);
      fprintf (stdout, ":\n");
   }
   format_indent (); 
}

void handle_keyword_do (char *s, char *token) {

   tokens_t tokens;
   create_token_list (&tokens, s, token);
   
   format_print ();

   if (tokens.count == 1) {
      fprintf (stdout, "while (1) {\n");
   } else {
      if (!strcmp (tokens.word[1], KEYWORD_WHILE)) {
         fprintf (stdout, "while (");
         eval_write (&tokens, 2, tokens.count - 1, false);
//       comp_write (&tokens, 2, tokens.count - 1);
         fprintf (stdout, ") {\n");
      } else if (!strcmp (tokens.word[1], KEYWORD_UNTIL)) {
         fprintf (stdout, "while (!(");
         eval_write (&tokens, 2, tokens.count - 1, false);
//       comp_write (&tokens, 2, tokens.count - 1);
         fprintf (stdout, ")) {\n");
      }
   }

   format_indent ();
}

void handle_keyword_loop (char *s, char *token) {

   tokens_t tokens;
   create_token_list (&tokens, s, token);

   if (tokens.count > 1) {
      if (!strcmp (tokens.word[1], KEYWORD_WHILE)) {
         format_print ();
         fprintf (stdout, "if (!(");
         eval_write (&tokens, 2, tokens.count - 1, false);
//       comp_write (&tokens, 2, tokens.count - 1);
         fprintf (stdout, ")) break;\n");
      } else if (!strcmp (tokens.word[1], KEYWORD_UNTIL)) {
         format_print ();
         fprintf (stdout, "if (");
         eval_write (&tokens, 2, tokens.count - 1, false);
//       comp_write (&tokens, 2, tokens.count - 1);
         fprintf (stdout, ") break;\n");
      }
   }

   format_outdent ();
   format_print ();
   fprintf (stdout, "}\n");
}

void handle_keyword_dim (char *s, char *token) {

   tokens_t tokens;

   create_token_list (&tokens, s, token);

   format_print ();

   int var_type_index = get_type_index (&tokens, 0);
   write_type (tokens.word[var_type_index]);
   int var_type = get_type (tokens.word[var_type_index]);
   int array_indices_count = 0;
   fprintf (stdout, " %s", tokens.word[1]);
   for (int i = 2; i <= var_type_index - 2; i++) {
      if (!strcmp (tokens.word[i], "(")) {
         fprintf (stdout, "[");
      } else if (!strcmp (tokens.word[i], ")")) {
         fprintf (stdout, "]");
      } else if (!strcmp (tokens.word[i], ",")) {
         fprintf (stdout, "][");
      } else {                                     // should be the array count
         fprintf (stdout, "%s", tokens.word[i]);
         array_indices_count++;
      }
   }
   fprintf (stdout, ";\n");

   entities_add (tokens.word[1], var_type, array_indices_count > 0, array_indices_count);
   entities_print ();
}

void handle_keyword_sub (char *s, char *token, bool is_def) {

   tokens_t tokens;

   create_token_list (&tokens, s, token);

   fprintf (stdout, "void %s ", tokens.word[1]);

   for (int i = 2; i < tokens.count; i++) {
      if (!strcmp (tokens.word[i], "(") ||
          !strcmp (tokens.word[i], ")")) {
         fprintf (stdout, "%s", tokens.word[i]);
      } else if (!strcmp (tokens.word[i], ",")) {
         fprintf (stdout, "%s ", tokens.word[i]);
      } else {
         int var_type_index = get_type_index (&tokens, i);
         write_type (tokens.word[var_type_index]);
         fprintf (stdout, " %s", tokens.word[i]);
         i += 2;
      }
   }

   if (is_def) {
      fprintf (stdout, ";\n");
   } else {
      fprintf (stdout, " {\n");
      entities_set_method_entry ();
   }

   format_indent ();
}

void handle_keyword_rem (char *s, char *token) {
   format_print ();
   while (is_whitespace (*s)) {                                   // skip whitespace
      s++;
   }
   fprintf (stdout, "// %s\n", s);
}

void handle_keyword_func (char *s, char *token, bool is_def) {

   tokens_t tokens;

   create_token_list (&tokens, s, token);

   if (!strcmp (tokens.word[tokens.count - 2], KEYWORD_AS)) {
      write_type (tokens.word[tokens.count - 1]);                  
      if (is_def) {
         int var_type = get_type (tokens.word[tokens.count - 1]);
         entities_add (tokens.word[1], var_type, false, 0);
      }
   } else {
      fprintf (stdout, "error: function return type not specified\n");
      exit (0);
   }

   fprintf (stdout, " %s ", tokens.word[1]);

   for (int i = 2; i < tokens.count - 2; i++) {
      if (!strcmp (tokens.word[i], "(") ||
          !strcmp (tokens.word[i], ")")) {
         fprintf (stdout, "%s", tokens.word[i]);
      } else if (!strcmp (tokens.word[i], ",")) {
         fprintf (stdout, "%s ", tokens.word[i]);
      } else {
         int var_type_index = get_type_index (&tokens, i);
         write_type (tokens.word[var_type_index]);
         fprintf (stdout, " %s", tokens.word[i]);
         i += 2;
      } 
   }

   if (is_def) {
      fprintf (stdout, ";\n");
   } else {
      fprintf (stdout, " {\n");
      entities_set_method_entry ();
   }

   entities_print ();

   format_indent ();
}

void eval_find_complete_expression (tokens_t *tokens, int start, int *end, int *eval_type) {
   int paren_count = 0;

   int eval_type_var;
   *eval_type = 0;

   *end = 0;
   for (int i = start; i < tokens->count; i++) {
      switch (tokens->word[i][0]) {
         case '(':
            paren_count++;
            break;
         case ')':
            paren_count--;
            if (paren_count < 0) {
               *end = i - 1;
               i = tokens->count - 1;  // force a break out of the for loop
            }
            break;
         case ';':
         case ',':
            if (!paren_count) {
               *end = i - 1;
               i = tokens->count - 1;  // force a break out of the for loop
            }
            break;
         default:

            if (!paren_count) {
               eval_type_var = entities_get_var_type (tokens->word[i]);    // process as if a variable
               *eval_type = max (*eval_type, eval_type_var);           // process as if a variable
               if (!eval_type_var) {
                  if (is_digit (tokens->word[i][0])) {                 // determine if this is a number
                     if (strstr (tokens->word[i], ".") ||              // check for special characters pushing this to a double
                         strstr (tokens->word[i], "e")) {
                        *eval_type = max (*eval_type, VARTYPE_DOUBLE);
                     } else {
                        *eval_type = max (*eval_type, VARTYPE_INT);    // default to integer. TBD: should assess the magnitude of the value
                                                                       // and assign accordingly.
                     }
                  } else if (tokens->word[i][0] == '"') {
                     *eval_type = max (*eval_type, VARTYPE_STRING);
                  }
               }
            }

            break;
      }
   }
   if (!*end) {
      *end = tokens->count - 1;
   }
}

void handle_keyword_open (char *s, char *token) {
   tokens_t tokens;
   create_token_list (&tokens, s, token);

   // open # id : name , mode
   // the name can be a string array element

   int id = atoi (tokens.word[2]);
   int end;
   int eval_type;
   eval_find_complete_expression (&tokens, 4, &end, &eval_type);

   format_print ();
   fprintf (stdout, "basic_file_open (%d, ", id);
   eval_write (&tokens, 4, end, false);
   fprintf (stdout, ", ");

   int i = end + 2;
   int m;
   if (!strcmp (tokens.word[i], KEYWORD_INPUT)) {
      m = 0;
   } else if (!strcmp (tokens.word[i], KEYWORD_OUTPUT)) {
      m = 1;
   } else if (!strcmp (tokens.word[i], KEYWORD_APPEND)) {
      m = 2;
   } else {
      fprintf (stdout, "open mode invalid\n");
      exit (0);
   }
   fprintf (stdout, "%d);\n", m);
}

void handle_keyword_close (char *s, char *token) {
   tokens_t tokens;
   create_token_list (&tokens, s, token);
              
   // close # id

   int id = atoi (tokens.word[2]);
   
   format_print ();
   fprintf (stdout, "basic_file_close (%d);\n", id);
}

void handle_keyword_delete (char *s, char *token) {
   tokens_t tokens;
   create_token_list (&tokens, s, token);

   format_print ();
   fprintf (stdout, "remove (");
   eval_write (&tokens, 1, tokens.count - 1, false);
   fprintf (stdout, ".val);\n");
}

void handle_keyword_return (char *s, char *token) {

   tokens_t tokens;
   create_token_list (&tokens, s, token);

   format_print ();

   fprintf (stdout, "return ");

   eval_write (&tokens, 1, tokens.count - 1, false);

   fprintf (stdout, ";\n");
}

void handle_keyword_else (char *s, char *token) {
   format_outdent ();
   format_print ();
   fprintf (stdout, "} else {\n");;
   format_indent ();
}

void handle_keyword_goto (char *s, char *token) {
   tokens_t tokens;
   create_token_list (&tokens, s, token);
   format_print ();
   fprintf (stdout, "goto %s%s;\n", LINE_LABEL_HEAD, tokens.word[1]);
}

void handle_keyword_if (char *s, char *token) {

   tokens_t tokens;

   create_token_list (&tokens, s, token);

   format_print ();

   fprintf (stdout, "if (");
   
   int then_index = get_index (&tokens, KEYWORD_THEN, 1);

   eval_write (&tokens, 1, then_index - 1, false);
// comp_write (&tokens, 1, then_index - 1);
/*
   for (int i = 1; i < then_index; i++) {
      if (!strcmp (tokens.word[i], TOKEN_EQUAL)) {
         fprintf (stdout, "==");
      } else if (!strcmp (tokens.word[i], TOKEN_NOT_EQUAL)) {
         fprintf (stdout, "!=");
      } else if (!strcmp (tokens.word[i], TOKEN_NOT)) {
         fprintf (stdout, "!");
      } else if (!strcmp (tokens.word[i], TOKEN_AND)) {
         fprintf (stdout, "&&");
      } else if (!strcmp (tokens.word[i], TOKEN_OR)) {
         fprintf (stdout, "||");
      } else {
         fprintf (stdout, "%s", tokens.word[i]);
      }
      if (i != then_index - 1) {
         fprintf (stdout, " ");
      }
   }
*/

   if (then_index <= tokens.count - 2) { // test that there are more tokens coming...
      fprintf (stdout, ") goto %s%s;", LINE_LABEL_HEAD, tokens.word[then_index + 1]);

      int else_index = get_index (&tokens, KEYWORD_ELSE, then_index);
      if (else_index) {
         fprintf (stdout, " else goto %s%s;", LINE_LABEL_HEAD, tokens.word[else_index + 1]);
      }
      fprintf (stdout, "\n");
   } else {
      fprintf (stdout, ") {\n");
      format_indent ();
   }
}

void handle_keyword_for (char *s, char *token) {

   tokens_t tokens;

   create_token_list (&tokens, s, token);

   format_print ();

   fprintf (stdout, "for (%s = ", tokens.word[1]);

   int token_initial_val_eval_start = 3;
   int token_initial_val_eval_end   = get_index (&tokens, KEYWORD_TO, token_initial_val_eval_start) - 1;
   eval_write (&tokens, token_initial_val_eval_start, token_initial_val_eval_end, false);
   fprintf (stdout, "; ");

   int token_stop_eval_start = token_initial_val_eval_start + 2;
   int token_stop_eval_end   = get_index (&tokens, KEYWORD_STEP, token_stop_eval_start);
   if (!token_stop_eval_end) {
      token_stop_eval_end = tokens.count - 1;
   } else {
      token_stop_eval_end--;
   }

   int token_step_eval_start = get_index (&tokens, KEYWORD_STEP, token_stop_eval_end);
   int token_step_eval_end;
   if (token_step_eval_start) {
      token_step_eval_start++;
      token_step_eval_end = tokens.count - 1;
   } else {
      strcpy (tokens.word[tokens.count], "1");
      tokens.count++;
      token_step_eval_start = tokens.count - 1;
      token_step_eval_end   = token_step_eval_start;
   }
   
   // (step > 0) ? (i <= b) : (i >= b)
   fprintf (stdout, "(");
   eval_write (&tokens, token_step_eval_start, token_step_eval_end, false);
   fprintf (stdout, " > 0) ? (");
   fprintf (stdout, "%s <= ", tokens.word[1]);
   eval_write (&tokens, token_stop_eval_start, token_stop_eval_end, false);
   fprintf (stdout, ") : (%s >= ", tokens.word[1]);
   eval_write (&tokens, token_stop_eval_start, token_stop_eval_end, false);
   fprintf (stdout, ")");

   fprintf (stdout, "; %s += ", tokens.word[1]);

   eval_write (&tokens, token_step_eval_start, token_step_eval_end, false);
   fprintf (stdout, ") {\n");

   format_indent ();
}

void handle_keyword_call (char *s, char *token) {
   
   tokens_t tokens;

   create_token_list (&tokens, s, token);

   format_print ();

   int start, end, eval_type;

   if (!strcmp (tokens.word[1], KEYWORD_GCHAR)) {                               // handle gchar 
      fprintf (stdout, "%s (", tokens.word[1]);
      start = 3; // skip over the subprogram name and the opening parenthesis
      eval_find_complete_expression (&tokens, start, &end, &eval_type);
      eval_write (&tokens, start, end, false);
      fprintf (stdout, ", ");
      start = end + 2; // skip over the comma
      eval_find_complete_expression (&tokens, start, &end, &eval_type);
      eval_write (&tokens, start, end, false);
      fprintf (stdout, ", &");
      start = end + 2; // skip over the comma
      eval_find_complete_expression (&tokens, start, &end, &eval_type);
      eval_write (&tokens, start, end, false);
      fprintf (stdout, ");\n");
   } else if (!strcmp (tokens.word[1], KEYWORD_POSITION)) {                            // handle sprite position
      fprintf (stdout, "%s (", tokens.word[1]);
      start = 3; // skip over the subprogram name and the opening parenthesis
      eval_find_complete_expression (&tokens, start, &end, &eval_type);
      eval_write (&tokens, start, end, false);
      fprintf (stdout, ", &");
      start = end + 2; // skip over the comma
      eval_find_complete_expression (&tokens, start, &end, &eval_type);
      eval_write (&tokens, start, end, false);
      fprintf (stdout, ", &");
      start = end + 2; // skip over the comma 
      eval_find_complete_expression (&tokens, start, &end, &eval_type);
      eval_write (&tokens, start, end, false);
      fprintf (stdout, ");\n"); 
   } else if (!strcmp (tokens.word[1], KEYWORD_SCREEN)) {
      fprintf (stdout, "console_border_color_set (");
      start = 3;
      eval_find_complete_expression (&tokens, start, &end, &eval_type);
      eval_write (&tokens, start, end, false);
      fprintf (stdout, ");\n");
   } else if (!strcmp (tokens.word[1], KEYWORD_FONTLOAD)) {
      fprintf (stdout, "console_font_load (");
      start = 3;
      eval_find_complete_expression (&tokens, start, &end, &eval_type);
      eval_write (&tokens, start, end, false);
      fprintf (stdout, ".val);\n");
   } else if (!strcmp (tokens.word[1], KEYWORD_CHARSET)) {
      fprintf (stdout, "console_fonts_load_std ();\n");
   } else if (!strcmp (tokens.word[1], KEYWORD_DISPLAY_MODE_GET)) {
      fprintf (stdout, "%s (&", tokens.word[1]);
      start = 3;
      eval_find_complete_expression (&tokens, start, &end, &eval_type);
      eval_write (&tokens, start, end, false);
      fprintf (stdout, ");\n");
   } else if (!strcmp (tokens.word[1], KEYWORD_DISPLAY_ROWS_AND_COLS)) {
      fprintf (stdout, "%s (&", tokens.word[1]);
      start = 3; // skip over the subprogram name and the opening parenthesis
      eval_find_complete_expression (&tokens, start, &end, &eval_type);
      eval_write (&tokens, start, end, false);
      fprintf (stdout, ", &");
      start = end + 2; // skip over the comma
      eval_find_complete_expression (&tokens, start, &end, &eval_type);
      eval_write (&tokens, start, end, false);
      fprintf (stdout, ");\n");
   } else if (!strcmp (tokens.word[1], KEYWORD_JOYST)) {
      fprintf (stdout, "%s (", tokens.word[1]);
      start = 3; // skip over the subprogram name and the opening parenthesis
      eval_find_complete_expression (&tokens, start, &end, &eval_type);
      eval_write (&tokens, start, end, false);
      fprintf (stdout, ", &");
      start = end + 2; // skip over the comma
      eval_find_complete_expression (&tokens, start, &end, &eval_type);
      eval_write (&tokens, start, end, false);
      fprintf (stdout, ", &");
      start = end + 2; // skip over the comma
      eval_find_complete_expression (&tokens, start, &end, &eval_type);
      eval_write (&tokens, start, end, false);
      fprintf (stdout, ");\n");
   } else if (!strcmp (tokens.word[1], KEYWORD_COINC)) {
      fprintf (stdout, "%s (", tokens.word[1]);
      start = 3; // skip over the subprogram name and the opening parenthesis
      eval_find_complete_expression (&tokens, start, &end, &eval_type);
      eval_write (&tokens, start, end, false);
      fprintf (stdout, ", ");
      start = end + 2; // skip over the comma
      eval_find_complete_expression (&tokens, start, &end, &eval_type);
      eval_write (&tokens, start, end, false);
      fprintf (stdout, ", ");
      start = end + 2; // skip over the comma
      eval_find_complete_expression (&tokens, start, &end, &eval_type);
      eval_write (&tokens, start, end, false);
      fprintf (stdout, ", &");
      start = end + 2; // skip over the comma
      eval_find_complete_expression (&tokens, start, &end, &eval_type);
      eval_write (&tokens, start, end, false);
      fprintf (stdout, ");\n");

   } else {
      eval_write (&tokens, 1, tokens.count - 1, false);
      fprintf (stdout, ";\n");
   }
}

void handle_keyword_input (char *s, char *token) {
   tokens_t tokens;
   create_token_list (&tokens, s, token);

   int colon_index = get_index (&tokens, ":", 1);
   int pound_index = get_index (&tokens, "#", 1);

   int file_id, start, end;

   if (!pound_index && !colon_index) {
      // regular input of a variable
      start   = 1;
      end     = tokens.count - 1;

      format_print ();
      fprintf (stdout, "string_input (&");
      eval_write (&tokens, start, end, false);
      fprintf (stdout, ");\n");

   } else if (!pound_index && colon_index) {

      // regular input after a prompt

      format_print ();
      fprintf (stdout, "string_print (stdout, ");
      eval_write (&tokens, 1, colon_index - 1, false);
      fprintf (stdout, ");\n");

      start   = colon_index + 1;
      end     = tokens.count - 1;

      format_print ();
      fprintf (stdout, "string_input (&");
      eval_write (&tokens, start, end, false);
      fprintf (stdout, ");\n");

   } else if (pound_index && colon_index) {

      // file input

      file_id = atoi (tokens.word[2]);
      start   = colon_index + 1;
      end     = tokens.count - 1;

      format_print ();
      fprintf (stdout, "basic_file_input (%d, &", file_id);
      eval_write (&tokens, start, end, false);
      fprintf (stdout, ");\n");

   } else {
      fprintf (stdout, "malformed input statement\n");
   }
}

void write_print (tokens_t *tokens, int start, int file_id);

void handle_keyword_display (char *s, char *token) {
   tokens_t tokens;
   create_token_list (&tokens, s, token);
   int ati;
   int atc;
   int eval_type;

   if ((ati = get_index (&tokens, KEYWORD_AT, 1))) {
      atc = get_index (&tokens, ":", ati + 1);

      format_print ();
      eval_write (&tokens, ati, atc - 1, false);
      fprintf (stdout, ";\n");

      write_print (&tokens, atc + 1, -1);
   } else {
      write_print (&tokens, 1, -1);
   }

   if (ati) {
      format_print ();
      fprintf (stdout, "atend ();\n");
   }
}

void handle_keyword_print (char *s, char *token) {

   tokens_t tokens;

   create_token_list (&tokens, s, token);

   if (!strcmp (tokens.word[1], "#")) {
      // print # id : stuff...
      // 0     1 2  3 4
      write_print (&tokens, 4, atoi (tokens.word[2]));
   } else {
      // print stuff...
      // 0     1
      write_print (&tokens, 1, -1);
   }
}

void write_print (tokens_t *tokens, int start, int file_id) {

   int end;
   int eval_type;

   char file_ids[32];
   if (file_id == -1) {
      strcpy (file_ids, "stdout");
   } else {
      sprintf (file_ids, "basic_file_get_stream (%d)", file_id);
   }

   bool write_cr;
   while (1) {
      write_cr = true;
      eval_find_complete_expression (tokens, start, &end, &eval_type);
      if (!end) {
         break;
      }
 
      #ifdef DEBUG
         fprintf (stdout, "// EVAL_TYPE=%d: start %d, end %d\n", eval_type, start, end);
      #endif

      format_print ();

      switch (eval_type) {
         case VARTYPE_STRING:
            fprintf (stdout, "string_print (%s, ", file_ids);
            break;
         case VARTYPE_INT:
            fprintf (stdout, "fprintf (%s, %s", file_ids, "\"\%");
            fprintf (stdout, "d\", ");
            break;
         case VARTYPE_LONG:
            fprintf (stdout, "fprintf (%s, %s", file_ids, "\"\%");
            fprintf (stdout, "ld\", ");
            break;
         case VARTYPE_FLOAT:
         case VARTYPE_DOUBLE:
            fprintf (stdout, "fprintf (%s, %s", file_ids, "\"\%");
            fprintf (stdout, "f\", ");
            break;
         default:
            fprintf (stdout, "error: type cannot be evaluated\n");
            exit (0);
            break;
      }

      eval_write (tokens, start, end, false);

      switch (eval_type) {
         case VARTYPE_STRING:
            fprintf (stdout, ");\n");
            break;
         case VARTYPE_INT:
         case VARTYPE_LONG:
         case VARTYPE_FLOAT:
         case VARTYPE_DOUBLE:
            fprintf (stdout, ");\n");
            break;
         default:
            break;
      }

      if (end < tokens->count - 1) {
         end++;
         if (!strcmp (tokens->word[end], ",")) {
            format_print ();
            fprintf (stdout, "fprintf (%s, \"\\t\");\n", file_ids);
            write_cr = false;
         } else if (!strcmp (tokens->word[end], ";")) {
            format_print ();
            fprintf (stdout, "fprintf (%s, \" \");\n", file_ids);
            write_cr = false;
         }
      }

      if (end == tokens->count - 1) {
         break;
      } 

      start = end + 1;
   }

   if (write_cr) {
      format_print ();
      fprintf (stdout, "fprintf (%s, \"\\n\");\n", file_ids);
   }
}

void handle_keyword_block_end () {
   format_print ();
   fprintf (stdout, "}\n");
}

void handle_keyword_subend () {
   format_outdent ();
   handle_keyword_block_end ();
   entities_set_method_exit ();
   entities_print ();
}

void handle_keyword_end (char *s, char *token) {

   tokens_t tokens;

   create_token_list (&tokens, s, token);

   if (tokens.count == 1) {
      format_print ();
      fprintf (stdout, "exit (0);\n");
   } else if (!strcmp (tokens.word[1], KEYWORD_IF)) {
      format_outdent ();
      handle_keyword_block_end ();
   } else if (!strcmp (tokens.word[1], KEYWORD_SELECT)) {
      format_outdent ();
      format_outdent ();
      handle_keyword_block_end ();
      select_case_level--;
   } else if (!strcmp (tokens.word[1], KEYWORD_SUB)) {
      handle_keyword_subend ();
   } else if (!strcmp (tokens.word[1], KEYWORD_FUNC)) {
      format_outdent ();
      handle_keyword_block_end ();
      entities_set_method_exit ();
      entities_print ();
   } else {
      fprintf (stdout, "error: unknown END type\n");
      exit (0);
   }
}

char *translate_line_keyword_start (char *s, char *token) {
   #ifdef DEBUG
      fprintf (stdout, "// keyword_start\n");
   #endif

   if (!strcmp (token, KEYWORD_DIM)) {
      handle_keyword_dim (s, token);
   } else if (!strcmp (token, KEYWORD_SUB)) {
      handle_keyword_sub (s, token, false);
   } else if (!strcmp (token, KEYWORD_FUNC)) {
      handle_keyword_func (s, token, false);
   } else if (!strcmp (token, KEYWORD_FOR)) {
      handle_keyword_for (s, token);
   } else if (!strcmp (token, KEYWORD_PRINT)) {
      handle_keyword_print (s, token);
   } else if (!strcmp (token, KEYWORD_DISPLAY)) {
      handle_keyword_display (s, token);
   } else if (!strcmp (token, KEYWORD_IF)) {
      handle_keyword_if (s, token);
   } else if (!strcmp (token, KEYWORD_SELECT)) {
      handle_keyword_select (s, token);
   } else if (!strcmp (token, KEYWORD_CASE)) {
      handle_keyword_case (s, token);
   } else if (!strcmp (token, KEYWORD_GOTO)) {
      handle_keyword_goto (s, token);
   } else if (!strcmp (token, KEYWORD_ELSE)) {
      handle_keyword_else (s, token);
   } else if (!strcmp (token, KEYWORD_SUBEND)) {
      handle_keyword_subend ();
   } else if (!strcmp (token, KEYWORD_NEXT)) {
      format_outdent ();
      handle_keyword_block_end ();
   } else if (!strcmp (token, KEYWORD_RETURN)) {
      handle_keyword_return (s, token);
   } else if (!strcmp (token, KEYWORD_END)) {
      handle_keyword_end (s, token);
   } else if (!strcmp (token, KEYWORD_CALL)) {
      handle_keyword_call (s, token);
   } else if (!strcmp (token, KEYWORD_REM) ||
              !strcmp (token, KEYWORD_REM_ALT)) {
      handle_keyword_rem (s, token);
   } else if (!strcmp (token, KEYWORD_DO)) {
      handle_keyword_do (s, token);
   } else if (!strcmp (token, KEYWORD_LOOP)) {
      handle_keyword_loop (s, token);
   } else if (!strcmp (token, KEYWORD_OPEN)) {
      handle_keyword_open (s, token);
   } else if (!strcmp (token, KEYWORD_CLOSE)) {
      handle_keyword_close (s, token);
   } else if (!strcmp (token, KEYWORD_DELETE)) {
      handle_keyword_delete (s, token);
   } else if (!strcmp (token, KEYWORD_INPUT)) {
      handle_keyword_input (s, token);
   } else {

      #ifdef DEBUG
         while (s) {
            fprintf (stdout, "[%s] ", token);
            s = get_token (s, token);
         }
         fprintf (stdout, "\n");
      #endif
   }
   return s;
}

char *translate_line_assignment (char *s, char *token) {
   #ifdef DEBUG
      fprintf (stdout, "// assignment\n");
   #endif

   tokens_t tokens;

   create_token_list (&tokens, s, token);

   format_print ();

   int assignment_index = get_index (&tokens, "=", 1);

   eval_write (&tokens, 0, assignment_index, true);
   fprintf (stdout, " ");

// for (int i = 0; i < 2; i++) {
//    fprintf (stdout, "%s ", tokens.word[i]);
// }
   eval_write (&tokens, assignment_index + 1, tokens.count - 1, false);

   fprintf (stdout, ";\n");
   return s;
}

char *translate_line (char *s, char *token) {
   if (is_keyword (token)) {
      translate_line_keyword_start (s, token);
   } else {
      translate_line_assignment (s, token);
   }
   return s;
}

char *process_subfuncs (char *s, char *token) {
   if (!strcmp (token, KEYWORD_SUB)) {
      handle_keyword_sub (s, token, true);
   } else if (!strcmp (token, KEYWORD_FUNC)) {
      handle_keyword_func (s, token, true);
   }
   return s;
}

void write_function_prototypes (const char *filename) {
   char s[LINE_LEN];
   char token[LINE_LEN];
   int line_num;
   char *t;
   
   FILE *f = fopen (filename, "r");
   if (f) {
      while (fgets (s, sizeof (s), f)) {
         s[strcspn (s, "!\r\n")] = 0x00;  // kill of CR and LF
 
         t = get_token (s, token);
         line_num = atoi (token);
         if (line_num) { 
            t = get_token (t, token);
         }
         if (strlen (token)) {
            t = process_subfuncs (t, token);
         }
      } 
      fclose (f);
   }
   fprintf (stdout, "\n");
}

void kill_bang_comments (char *s) {
   bool in_string = false;          // whether in a string
   char *p = s;                     // pointer within s
   bool cont = true;                // whether done
   while (cont) {                   // loop until done
      switch (*p) {                 // eval this character
         case '"':                  // handle quotes
            in_string = !in_string; // update whether a string has started or ended
            break;
         case '!':                  // handle bang
            if (!in_string) {       // if not in a string this is a comment
               *p   = 0x00;         // null terminate
               cont = false;        // exit the loop
            }
            break;
         case 0:
            cont = false;           // exit the loop
            break;
      }
      p++;                          // move to the next position
   }
}

void translate_file (const char *filename) {
   char s[LINE_LEN];
   char token[LINE_LEN];
   int line_num;
   char *t;

   FILE *f = fopen (filename, "r");
   if (f) {
      while (fgets (s, sizeof (s), f)) {
         s[strcspn (s, "\r\n")] = 0x00;  // kill of CR and LF

         kill_bang_comments (s);

         #ifdef DEBUG
            fprintf (stdout, "// %s\n", s);
         #endif
         string_convert_non_literals_to_lower_case (s);

         t = get_token (s, token);
         line_num = atoi (token);
         if (line_num) {                                               // write C label
            fprintf (stdout, "%s%d:\n", LINE_LABEL_HEAD, line_num);
            t = get_token (t, token);
         }
         if (strlen (token)) {
            t = translate_line (t, token);
         } else {
            fprintf (stdout, "\n");
         }
      } 
      fclose (f);
   }
}

void write_main () {
   fprintf (stdout, "\n");
   fprintf (stdout, "int main (int argc, char *argv[]) {\n");
   fprintf (stdout, "\n");

   fprintf (stdout, "   %s = atof (\"3.1415926535898\");\n", KEYWORD_PI);
   fprintf (stdout, "\n");

   fprintf (stdout, "   basic_files_init ();\n");
   fprintf (stdout, "\n");

   fprintf (stdout, "   %s ();\n", BMAIN_METHOD);
   fprintf (stdout, "\n");

   fprintf (stdout, "   return 0;\n");
   fprintf (stdout, "}\n");
}

bool file_exists (const char *filename) {
   bool r = false;
   FILE *f = fopen (filename, "r");
   if (f) {
      fclose (f);
      r = true;
   }
   return r;
}

int main (int argc, char *argv[]) {

   char filename[FILENAME_LEN];
   
   if (argc == 2) {
      strcpy (filename, argv[1]);

      if (file_exists (filename)) {
         entities_init ();
         write_includes ();        
         write_built_in_basic_methods ();
         entities_print ();
         write_function_prototypes (filename);
         format_init ();
         translate_file (filename);
         write_main ();
      }
   } else {
      fputs ("bc: usage: bc filename\n", stderr);
   }

   return 0;
}
