/* color.c
   This program sets the foreground and background color.
   Usage: color [-f color1] [-b color2]
   Options:
   -b set background color
   -d set default ti99 foreground and background color
   -f set foreground color
   -w set to white on black
   -x set to black on white
   change log:
   08/24/2023 initial version
   09/22/2023 handle case where spurious non-option inputs are supplied
   09/22/2023 modified to require -d to set the default color
   10/22/2023 added -w and -x options
   10/26/2023 added -w and -x to the usage statement
   11/01/2023 updated to use console_display_get_mode rather than reading directly from the private cache
              updated to use console_text_get_foreground_color and console_text_get_background_color
              removed direct use of cache
   11/03/2023 removed reference to vdp.h
   12/10/2023 removed reference to string_ext.h
   02/27/2024 added use of dylib
   03/01/2024 updates to use string constants with dylib.fputs
   12/27/2024 updated the default text color
   01/20/2025 moved dylib.fputs to dylib
   02/28/2025 adjusted for removal of 80x30 display mode
   06/15/2025 updated exit references
*/

#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <stdbool.h>
#include <stdio.h>
#include <console.h>
#include <dylib.h>

// prints usage info
const char *usage = "usage: color [-d] [-w] [-x] [-f color1] [-b color2]\n";
void print_usage () {
   dylib.fputs (usage, stdout);
}

// the color program
const char *msg_illegal_color = "color: illegal color\n";
int main (int argc, char *argv[]) {
   unsigned int fcolor = console_text_get_foreground_color ();   // capture the current foreground color
   unsigned int bcolor = console_text_get_background_color ();   // capture the current background color
   bool error          = false;                                  // error flag
   bool opt_set        = false;                                  // indicates if at least one option was set

   // capture the options
   int opt;
   while ((opt = getopt (argc, argv, "df:b:wx")) != -1) {
      switch (opt) {                            
         case 'd':                                               // handle the default color
            fcolor  = COLOR_DEFAULT_TEXT_FOREGROUND;
            bcolor  = COLOR_DEFAULT_TEXT_BACKGROUND;           
            opt_set = true;
            break;
         case 'w':
            fcolor  = COLOR_WHITE;
            bcolor  = COLOR_BLACK;
            opt_set = true;
            break;
         case 'x':
            fcolor  = COLOR_BLACK;
            bcolor  = COLOR_WHITE;
            opt_set = true;
            break;
         case 'f':                                               // handle the foreground color
            fcolor = atoi (optarg);
            if (fcolor < 0 || fcolor > 15) {                     // verify the color is in range
               dylib.fputs (msg_illegal_color, stderr);
               error = true;
            }
            opt_set = true;
            break; 
         case 'b':                                               // handle the background color
            bcolor = atoi (optarg);
            if (bcolor < 0 || bcolor > 15) {                     // verify the color is in range
               dylib.fputs (msg_illegal_color, stderr);
               error = true;
            }
            opt_set = true;
            break; 
         case '?':                                               // handle unknown options as errors
         default:
            error = true;       
            break;
      }
   }

   // adjust argc and argv removing the processed options
   argc -= optind;
   argv += optind;

   if (error || argc || !opt_set) {                              // exit if there was an error while processing options, no options
      print_usage ();                                            // were set or there are extra parameters
      exit (0);
   }

   console_text_set_foreground_color (fcolor);
   console_text_set_background_color (bcolor);
                                                                 // in text mode but background is for text and color modes.

   int mode;
   int rows;
   console_display_get_mode (&mode, &rows);
   if (mode == DISPLAY_MODE_STANDARD) {                          // if standard (color) mode then set the color of all char sets
      console_standard_set_all_colors (fcolor, bcolor);
   }           

   return 0;                                                     // done processing, return
}
