/* cp.c
   This program implements the copy command.
   limitations
   - implements only the -i, -n and -v options (the most common, and even then rarely needed)
   change history
   05/31/2025 initial version, adapted from mv.c
   06/02/2025 added recursive copy option
   06/02/2025 added verbosity
   06/10/2025 moved temp file to /tmp
*/

#include <utility_private.h>
#include <batch_private.h>
#include <dylib.h>
#include <unistd.h>
#include <unistd_private.h>

// displays program usage
void display_usage () {
   dylib.fputs ("usage: cp [-i | -n] [-v] source target\n", stderr);
   dylib.fputs ("       cp [-i | -n] [-R | -r] [-v] source ... directory\n", stderr);
}

#define BATCH_NAME "/tmp/cp_t"

// the main procedure
int main (int argc, char *argv[]) {

   bool interactive_overwrite = false;                        // option flags
   bool overwrite_files       = true;
   bool copy_recursively      = false;
   bool verbose               = false;
   bool debug                 = false;
   bool error                 = false;                        // indicates whether option processing resulted in an erro

   int opt;
   while ((opt = getopt (argc, argv, "inRrvd")) != -1) {      // capture the options
      switch (opt) {
         case 'i':                                            // interactive overwrite
            overwrite_files       = false;
            interactive_overwrite = true;
            break;
         case 'n':                                            // prevent overwrite
            overwrite_files       = false;
            interactive_overwrite = false;
            break;
         case 'R':                                            // recursive copy
         case 'r':
            copy_recursively      = true;
            break;
         case 'v':                                            // verbosity
            verbose               = true;
            break;
         case 'd':
            debug                 = true;
            break;
         case '?':                                            // unrecognized option
         default:                                             // error
            error = true;                      
            break;
      }
   }

   int r = 0;                                                 // initial return result
   int r2;

   if (!error) {                                              // continue executing if option processing succeeded

      argc -= optind;                                         // adjust argc and argv removing the processed options
      argv += optind;
   
      if (argc >= 2) {                                        // ensure there are at least two parameters
         if (argc == 2 || is_dir (argv[argc - 1])) {          // confirm POSIX case 1 or case 2
            batch_t batch = batch_create (BATCH_NAME, verbose, verbose, debug);
            for (int i = 0; i < argc - 1; i++) {              // process all the parameters
               if (!dsr_path_is_root (argv[i])) {             // ensure parameter isn't a mounted filesystem
                  r2 = copy_with_remove                       // perform the operation
                          ("cp",
                           false,
                           copy_recursively,
                           &batch,
                           argv[i], 
                           argv[argc - 1], 
                           overwrite_files, 
                           interactive_overwrite, 
                           verbose);
                  r += r2;
               } else {                                       // write diagnostic message re mounted paths
                  dylib.fputs ("cp: ", stderr);
                  dylib.fputs (argv[i], stderr);
                  dylib.fputs (": cannot move mounted path\n", stderr);
                  r = 1;
               }
            }
            batch_close (&batch);
            batch_execute (&batch);
            batch_remove (&batch);
         } else {                                             // handle the POSIX option 2 case failed (last arg doesn't exist or isn't a directory
            dylib.fputs ("cp: ", stderr);
            dylib.fputs (argv[argc - 1], stderr);
            dylib.fputs (" is not a directory\n", stderr);
            r = 1;
         }
      } else {
         display_usage ();                                    // display usage
         r = 1;                                               // set to failed
      }

   } else {
      display_usage ();                                       // display usage since there was an error processing options
      r = 1;
   }

   return r;                                                  // return the result
}
