/* df.c
   This program displays free disk space.
   Usage: df filesystem1 [... filesystemn]
   change log:
   06/23/2023 initial version
   08/05/2023 added support for cwd
   11/21/2023 updated to use FILENAME_MAX
   03/07/2024 updated to use dylib
   01/20/2025 moved fputs to dylib
   06/08/2025 added getopt to prevent options from being processed as paths
   06/15/2025 updated exit references
*/

#include <stdbool.h>
#include <stdio.h>
#include <unistd.h>
#include <statfs.h>
#include <dylib.h>
#include <stdlib.h>

void diskfree_list (const char *path) {
   static bool display_titles = true;
   struct statfs sfs;
   int r = statfs (path, &sfs);
   if (!r) {
      if (display_titles) {
         fprintf (stdout, "%-11s %7s %7s %7s\n", "volume", "total", "used", "avail");
         display_titles = false;
      }
      fprintf (stdout, "%-11s %7.0f %7.0f %7.0f\n", sfs.volume, sfs.f_blocks, sfs.f_blocks - sfs.f_bfree, sfs.f_bfree);
   } else {
      fprintf (stderr, "df: %s: No such file or directory\n", path);
   }
}

void display_usage () {
   dylib.fputs ("usage: df [path ...]\n", stderr);
}

int main (int argc, char *argv[]) {

  // capture the options
   int error = false;
   int opt;
   while ((opt = getopt (argc, argv, "")) != -1) {
      switch (opt) {
         case '?':
         default:
            error = true;
            break;
      }
   }

   if (error) {
      display_usage ();
      exit (1);
   }

   // adjust argc and argv removing the processed options
   argc -= optind;
   argv += optind;

   // determine if a set of paths have been provided or cwd should be used
   if (argc) {
      for (int i = 0; i < argc; i++) {              // write stats for all the requested file systems
         diskfree_list (argv[i]);
      }
   } else {
      char cwd[FILENAME_MAX];
      char *p = dylib.getcwd (cwd, sizeof (cwd));   // get the cwd
      if (p) {
         diskfree_list (cwd);                       // write the stats
      } else {
         dylib.fputs ("df: cwd not set\n", stdout); // output an error
      }
   }
 
   return 0;
}
