/* display.c
   This program sets the system display mode.
   Usage: color [-f color1] [-b color2]
   Options:
   -f set to the F18A 24x80 column text mode
   -s set to standard mode (color 24x32)
   -t set to text mode (24x40)
   change log:
   11/01/2023 initial version
   02/27/2024 updated to use dylib
   12/01/2024 added -q option
   01/20/2025 updated fputs to use dylib
   02/02/2025 added F18A text 80x30 mode
              added detailed / attribute printing
   02/28/2025 adjusted for remove of 80x30 display mode
   06/16/2025 updated exit reference
   10/03/2025 added validation of rows as the boot values are initialized to 0 instead of 24/30
*/

#include <constants.h>
#include <stdbool.h>
#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <console.h>
#include <dylib.h>

// prints usage info
void print_usage () {
   dylib.fputs ("usage: display { [-s] [-t] [-f] [-2] [-3] [-q] [-Q]}\n", stdout);
}

// the display program
int main (int argc, char *argv[]) {

   bool error                   = false;       // error flag
   bool display_status          = false;
   bool display_detailed_status = false;
   bool settings_changed        = false;
   int requested_mode;
   int requested_rows;
   
   console_display_get_mode (&requested_mode, &requested_rows);

   // ensure an appropriate value is used
   if (requested_rows != DISPLAY_ROWS_24 &&
       requested_rows != DISPLAY_ROWS_30) {
      requested_rows = DISPLAY_ROWS_DEFAULT;
   }

   // capture the options
   int opt;
   while ((opt = getopt (argc, argv, "23fstqQ")) != -1) {
      switch (opt) {                            
         case 'f': // handle the f18a text mode      
            requested_mode   = DISPLAY_MODE_F18A_TEXT80;
            settings_changed = true;
            break;
         case 's': // handle standard mode
            requested_mode   = DISPLAY_MODE_STANDARD;
            settings_changed = true;
            break;
         case 't': // handle text mode
            requested_mode   = DISPLAY_MODE_TEXT;
            settings_changed = true;
            break;
         case '2': // handle rows
            requested_rows   = DISPLAY_ROWS_24;
            settings_changed = true;
            break;
         case '3': // handle rows
            requested_rows   = DISPLAY_ROWS_30;
            settings_changed = true;
            break;
         case 'q': // display the current mode
            display_status = true;
            break;
         case 'Q': // display the current mode
            display_detailed_status = true;
            break;
         case '?': // handle unknown options as errors
         default:
            error = true;       
            break;
      }
   }

   // adjust argc and argv removing the processed options
   argc -= optind;
   argv += optind;

   // exit if there was an error while processing options
   if (error || argc) {
      print_usage ();                                               // were set or there are extra parameters
      exit (0);
   }

   // set the display mode or rows changed
   if (settings_changed) {
      unsigned int fcolor = console_text_get_foreground_color ();   // capture the current foreground color
      unsigned int bcolor = console_text_get_background_color ();   // capture the current background color

      // if the requested mode is different than the current mode, then change the mode and clear the screen
      console_display_set_mode (requested_mode, requested_rows);

      switch (requested_mode) {
         case DISPLAY_MODE_STANDARD:
            console_border_color_set (bcolor);
            console_standard_set_all_colors (fcolor, bcolor);
            break;
         case DISPLAY_MODE_TEXT:
         case DISPLAY_MODE_F18A_TEXT80:
            console_text_set_foreground_color (fcolor);
            console_text_set_background_color (bcolor);
            break;
         default:
            break;
      }
      console_cls ();
   }

   // display the mode if requested
   if (display_status) {
      switch (requested_mode) {
         case DISPLAY_MODE_STANDARD:
            dylib.fputs ("standard\n", stdout);
            break;
         case DISPLAY_MODE_TEXT:
            dylib.fputs ("text_40\n", stdout);
            break;
         case DISPLAY_MODE_F18A_TEXT80:
            dylib.fputs ("f18_text_80\n", stdout);
            break;
            break;
         default:
            break;
      }
   }

   // display the detailed status if requested
   if (display_detailed_status) {
      switch (requested_mode) {
         case DISPLAY_MODE_STANDARD:
            dylib.fputs ("mode:             standard\n", stdout);
            dylib.fputs ("chardef geometry: 8 x 8\n", stdout);
            dylib.fputs ("colors:           15 fg/bg per 8\n", stdout);
            break;
         case DISPLAY_MODE_TEXT:
            dylib.fputs ("mode:             text\n", stdout);
            dylib.fputs ("chardef geometry: 8 x 6\n", stdout);
            dylib.fputs ("colors:           2 (monochrome)\n", stdout);
            break;
         case DISPLAY_MODE_F18A_TEXT80:
            dylib.fputs ("mode:             F18A text 80\n", stdout);
            dylib.fputs ("chardef geometry: 8 x 6\n", stdout);
            dylib.fputs ("colors:           2 (monochrome)\n", stdout);
            break;
         default:
            break;
      }
      fprintf (stdout, "screen geometry:  %d x %d\n", console_display_get_rows (), console_display_get_columns ());
   }

   return 0;                                                     // done processing, return
}
