/* double.c
   This program tests the use of the double type.
   change log:
   11/15/2023 initial version
   01/20/2025 updated fputs to dylib
*/

#include <stdlib.h>
#include <conversion.h>
#include <stdio.h>
#include <string.h>
#include <assert.h>
#include <dylib.h>

void double_assignment () {
   fprintf (stdout, "double assignment\n\n");
   
   double d;
   int p = 0; 
   char *dv[] = {
      "1.020304050607",
      "10.20304050611",
      "0.5",
      "-10.20304050618",
      "-100.0", 
      ".", 
      "-", 
      "1",
      "1.5",
      "1500", 
      "9900",
      "9999.00",
      "-9900.01",
      "1.0E0",
      "7.01E1",
      "70.1",
      "7.01e-1",
      "0.701",
      "0.01",
      "1E126",
      NULL
   };

   while (dv[p]) {
      d = atof (dv[p]);
      fprintf (stdout, "%15s %16.9f\n", dv[p], d);
      p++;
   }

   fprintf (stdout, "%18s %8p", "ptr to double", (void *) &d);
}

void double_addition () {
   const char *d1s[] = {"1.020304",     "1.020304", "1.0",  "9.0",     "8.9999", ""};
   const char *d2s[] = {"0.0102060405", "9.0",      "9.0", "-8.9999", "-8.9999", ""};
   double d1, d2, ds;

   dylib.fputs ("double addition\n\n", stdout);

   int i = 0;
   while (strlen (d1s[i]) != 0) {
      d1 = atof (d1s[i]);
      d2 = atof (d2s[i]);
      ds = d1 + d2;
      fprintf (stdout, "%8p %18.10f + \n", (void *) &d1, d1);
      fprintf (stdout, "%8p %18.10f = \n", (void *) &d2, d2);
      fprintf (stdout, "%8p %18.10f\n\n", (void *) &ds, ds);
      i++;
   }
}

void double_subtraction () {

   const char *d1s[] = { "8.0", "10.203040506",   "10.0",  "9.9999", "8.9999", NULL};
   const char *d2s[] = {"10.0",  "9.102030405", "1010.0", "10.0000", "8.9999", NULL};
   
   double d1, d2, ds;
   
   fprintf (stdout, "double subtraction\n\n");
   
   int i = 0;
   while (d1s[i]) {
      d1 = atof (d1s[i]);
      d2 = atof (d2s[i]);
      ds = d1 - d2;
      fprintf (stdout, "%8p %18.10f - \n", (void *) &d1, d1);
      fprintf (stdout, "%8p %18.10f = \n", (void *) &d2, d2);
      fprintf (stdout, "%8p %18.10f\n\n", (void *) &ds, ds);
      i++;
   }
}

void double_multiplication () {

   const char *d1s[] = {"10.0", "1.0", "10.0",  "9.9999", "8.9999", NULL};
   const char *d2s[] = { "8.0", "1.0", "11.0", "10.0000", "8.9999", NULL};
   double d1, d2, ds;
   
   fprintf (stdout, "double multiplication\n\n");
   
   int i = 0;
   while (d1s[i]) {
      d1 = atof (d1s[i]);
      d2 = atof (d2s[i]);
      ds = d1 * d2;
      fprintf (stdout, "%8p %18.10f * \n", (void *) &d1, d1);
      fprintf (stdout, "%8p %18.10f = \n", (void *) &d2, d2);
      fprintf (stdout, "%8p %18.10f\n\n", (void *) &ds, ds);
      i++;
   }
}     

void double_division () {

   const char *d1s[] = {"10.0", "10.0", "20.0",  "9.9999", "8.9999", NULL};
   const char *d2s[] = {"10.0",  "1.0", "11.0", "10.0000", "8.9999", NULL};
   double d1, d2, ds;
   
   fprintf (stdout, "double division\n\n");
      
   int i = 0;
   while (d1s[i]) {
      d1 = atof (d1s[i]);
      d2 = atof (d2s[i]);
      ds = d1 / d2;
      fprintf (stdout, "%8p %18.10f / \n", (void *) &d1, d1);
      fprintf (stdout, "%8p %18.10f = \n", (void *) &d2, d2);
      fprintf (stdout, "%8p %18.10f\n\n", (void *) &ds, ds);
      i++;
   }  
}     

void double_math () {
   dylib.fputs ("double math\n\n", stdout);

   double d1, d2, ds;

   d1 = atof ("8.0");
   d2 = atof ("10.0");
   ds = d1 - d2;
   fprintf (stdout, "%8p %18.10f - \n", (void *) &d1, d1);
   fprintf (stdout, "%8p %18.10f = \n", (void *) &d2, d2);
   fprintf (stdout, "%8p %18.10f\n\n", (void *) &ds, ds);

   d1 = ds;
   d2 = ds;
   ds = d1 * d2;
   fprintf (stdout, "%8p %18.10f * \n", (void *) &d1, d1);
   fprintf (stdout, "%8p %18.10f = \n", (void *) &d2, d2);
   fprintf (stdout, "%8p %18.10f\n\n", (void *) &ds, ds);

   // sadly our GCC implementation doesn't know how to negate correctly
   // d1 = -d1;
   d1 = fnegate (d1);
   d2 = ds;
   ds = d1 * d2;
   fprintf (stdout, "%8p %18.10f * \n", (void *) &d1, d1);
   fprintf (stdout, "%8p %18.10f = \n", (void *) &d2, d2);
   fprintf (stdout, "%8p %18.10f\n\n", (void *) &ds, ds);

   int i;

   i  = 0;
   d1 = itof (i);
   fprintf (stdout, "%d %8p %10.2f\n", i, &d1, d1);

   i  = 10;
   d1 = itof (i);
   fprintf (stdout, "%d %8p %10.2f\n", i, &d1, d1);

   i  = 32767;
   d1 = itof (i);
   fprintf (stdout, "%d %8p %10.2f\n", i, &d1, d1);

   i  = -1;
   d1 = itof (i);
   fprintf (stdout, "%d %8p %10.2f\n", i, &d1, d1);

   i  = -32767;
   d1 = itof (i);
   fprintf (stdout, "%d %8p %10.2f\n", i, &d1, d1);

   unsigned int ui;

   ui = (unsigned int) 0;
   d1 = uitof (ui);
   fprintf (stdout, "%u %8p %10.2f\n", ui, &d1, d1);

   ui = (unsigned int) 10;
   d1 = uitof (ui);
   fprintf (stdout, "%u %8p %10.2f\n", ui, &d1, d1);

   ui = (unsigned int) 63 * 1024 + 1023;
   d1 = uitof (ui);
   fprintf (stdout, "%u %8p %10.2f\n", ui, &d1, d1);
}

void double_math_and_cmp () {
   double d2 = atof ("2.0");
   double d1 = d2;
   int i;
   
   dylib.fputs ( __func__, stdout);
   dylib.fputs ("\n\n", stdout);
   
   // walk up and down with mult and div, and equ comparison
   for (i = 0; i < 16; i++) {
      d1 *= d2;
   }  
   for (i = 0; i < 16; i++) {
      d1 /= d2;
   }
   assert (d1 == d2);
   
   // walk up and down with add and sub, and equ comparison
   for (i = 0; i < 16; i++) {
      d1 += d2;
   }
   for (i = 0; i < 16; i++) {
      d1 -= d2;
   }
   assert (d1 == d2);
   
   // comparisions with 1 and 2
   d1 /= d1; // d1 should now be 1.0
   assert (d1 < d2);
   assert (d2 > d1);
   assert (d1 <= d2);
   assert (d2 >= d1);

   // comparisions with 2 and 2
   d1 = d2;
   assert (!(d1 > d2)); 
   assert (!(d1 < d2));
   assert (d1 <= d2);
   assert (d1 >= d2);
   
   // comparisons with -1 and 2
   d1 /= d1; // d1 = 1
   d1 = -d1; // d1 = -1
   assert (d1 < d2);
   assert (d2 > d1);
   assert (d1 <= d2);
   assert (d2 >= d1);
   
   // comparisions with -2 and -2
   d2 = -d2; // d2 = -2
   d1 = d2;
   assert (!(d1 < d2));
   assert (!(d2 > d1));
   assert (d1 <= d2);
   assert (d2 >= d1);
}  

int main (int argc, char *argv[]) {
   double_assignment ();
   double_math ();
   double_math_and_cmp ();
   double_addition ();
   double_subtraction ();
   double_multiplication ();
   double_division ();
   return 0;
}
