/* file_binary.c
   This program tests binary file support.
   change log:
   12/20/2023 initial version
   02/27/2024 added use of dylib
   01/10/2025 updated to use dylib.fwrite
   01/20/2025 moved fputs to dylib
              added removal of generated file
   07/05/2025 dylib adjustments
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <conversion.h>
#include <math.h>
#include <constants.h>
#include <assert.h>
#include <dylib.h>

void write_fstats (FILE *f) {
   dylib.fputs ("len ", stdout);
   dylib.fputs (dylib.int2str (f->bstate.len), stdout); 
   dylib.fputs (", pos ", stdout);
   dylib.fputs (dylib.int2str (f->bstate.pos), stdout); 
   dylib.fputs (", lb ", stdout);
   dylib.fputs (dylib.int2str (f->bstate.last_block), stdout); 
   dylib.fputs ("\n", stdout);
}

int main (int argc, char *argv[]) {

#define COUNT 10
#define LEN   40
#define FILENAME "binary"

   char s[COUNT][LEN];

   FILE *f;
   int i;
   int r;

   // WRITE A FILE

   // open the file
   f = dylib.fopen (FILENAME, "wb");
   assert (f);
   write_fstats (f);

   // initialize the data to be written
   memset (s, 0, sizeof (s));
   for (i = 0; i < COUNT; i++) {
      dylib.strcpy (s[i], "Your Rec ");
      dylib.strcat (s[i], dylib.int2str (i));
   }

   // write the data
   r = dylib.fwrite (s, LEN, COUNT, f);  
   assert (r == COUNT);
   write_fstats (f);

   // seek to the absolute file position 0x1000
   r = dylib.fseek (f, 0x1000, SEEK_SET);
   assert (!r);
   write_fstats (f);

   // update the data to be written in this second location
   memset (s, 0, sizeof (s));
   for (i = 0; i < COUNT; i++) {
      dylib.strcpy (s[i], "My Rec ");
      dylib.strcat (s[i], dylib.int2str (i));
   }

   // write the next data, but since the fseek positioned beyond the physical end, this will write empty blocks between the
   // current physical end and this position
   r = dylib.fwrite (s, LEN, COUNT, f);  
   assert (r == COUNT);
   write_fstats (f);

   // close the file
   r = dylib.fclose (f);
   assert (!r);


   // READ FROM THE FILE

   // initialize the memory
   memset (s, 0, sizeof (s));

   // open the file
   f = dylib.fopen (FILENAME, "rb");
   assert (f);
   write_fstats (f);

   // read the data
   r = dylib.fread (s, LEN, COUNT, f);
   assert (r == COUNT);
   write_fstats (f);

   // print the data
   for (i = 0; i < COUNT; i++) {
      dylib.fputs (s[i], stdout);
      dylib.fputs ("\n", stdout);
   }

   // seek to position 0x1000 in the file
   r = dylib.fseek (f, 0x1000, SEEK_SET);
   assert (!r);
   write_fstats (f);

   // read the second set of data
   r = dylib.fread (s, LEN, COUNT, f);
   assert (r == COUNT);

   // display the data
   for (i = 0; i < 10; i++) {
      dylib.fputs (s[i], stdout);
      dylib.fputs ("\n", stdout);
   }
 
   // close the file
   r = dylib.fclose (f);
   assert (!r);

   // remove the file
   r = remove (FILENAME);
   assert (!r);

   return 0;
}
