/* fontload.c
   This program loads fonts from one or more files. Each file can include one or more fonts, each
   assigned to an ASCII value. The fonts in any file, including the same file, can have duplicate
   ASCII assignments.
   Usage: fontload [-d] [[file [... filen]]
   change log:
   08/11/2023 initial version
   08/20/2023 modified to use dylib.fputs rather than fprintf
   09/02/2023 added use of getopt and ability to load std fonts with -d
   02/27/2024 updated to use dylib
   01/20/2025 updated fputs to dylib
   06/16/2025 updated exit reference
*/

#include <unistd.h>
#include <console.h>
#include <stdio.h>
#include <stdlib.h>
#include <dylib.h>

// prints usage info
void print_usage () {
   dylib.fputs ("usage: fontload [-d] [font_path1 [... font_pathn]\n", stdout);
}

int main (int argc, char *argv[]) {

   // capture the options
   int opt;
   bool error              = false;                              // error flag
   bool load_default_fonts = false;                              // load default fonts flag
   while ((opt = getopt (argc, argv, "d")) != -1) {
      switch (opt) {
         case 'd':                                               // handle loading default fonts
            load_default_fonts = true;
            break;
         case '?':                                               // handle unknown options as errors
         default:
            error = true;
            break;
      }
   }

   // adjust argc and argv removing the processed options
   argc -= optind; 
   argv += optind; 

   if (error ||                                                  // exit if there was an error processing options, or neither
       ((!load_default_fonts) &&                                 // default fonts were requested or other fonts        
        (argc == 0))) {
      print_usage ();
      exit (0);
   }

   if (load_default_fonts) {                                     // set the default fonts if requested
      console_fonts_load_std ();
   }

   int i, r;

   for (i = 0; i < argc; i++) {
      r = console_font_load (argv[i]);
      if (r) {
         dylib.fputs ("fontload: ", stderr);
         dylib.fputs (argv[i], stderr);
         dylib.fputs (": No such file or directory\n", stderr);
      }
   }

   return 0;
}
