/* grep.c
   This program is a file pattern searcher.
   Usage: grep [-i] [-v] [-w] pattern [file [... filen]]
   Options:
   -i case insensitivity
   -v invert the search
   -w word search
   change log:
   06/23/2023 initial version
   07/05/2023 added case-insensitive search, word search
   07/10/2023 added error message for files that cannot be opened
   08/21/2023 replaced use of fprintf with dylib.fputs to conserve memory
   12/10/2023 removed reference to string_ext.h
   02/27/2024 updated to use dylib
   01/20/2025 moved fputs to dylib
*/

#include <stdbool.h>
#include <string.h>
#include <assert.h>
#include <stdio.h>
#include <unistd.h>
#include <dylib.h>

// returns true if the provided character is typically used in a word (numbers, letters and underscore)
bool is_word_char (int c) {
   return (c == '_') ||
          ((c >= '0') && (c <= '9')) ||
          ((c >= 'A') && (c <= 'Z')) ||
          ((c >= 'a') && (c <= 'z'));
}

// tests the result of strstr and strcasestr to determine if what was found is a whole word and not part of a
// greater word (found "in" in "walking")
bool test_for_word (const char *haystack, const char *needle, const char *found) {
   int len = dylib.strlen (needle);        // get the length of the search word
   char *word_prev = (char *) found - 1;   // capture a pointer to character before the found word
   char *word_next = (char *) found + len; // advance beyond the found word
   bool wb, wf, wl;
   wb = found == haystack;                 // assess whether the found word is at the beginning of the line
   if (word_prev < haystack) {             // test if the word prev is before the haystack, as it would be an invalid test
      wf = false;                          // ok, is not valid, so set false
   } else {
      wf = !is_word_char (*word_prev);     // test the prev character
   }
   wl = !is_word_char (*word_next);        // test the next character
   return (wb || wf) && wl;                // return the result
}

// tests whether the haystack has the needle word, ignoring or considering case as indicated
bool has_word (const char *haystack, const char *needle, bool ignore_case) {
   char *k;
   k = (char *) haystack;
   bool r = false;
   while (k) {                                     // search through all of the haystack as the needle might be contained in
                                                   // the contents of other words before the exact word is found
      if (ignore_case) {                           // perform the requested search
         k = strcasestr (k, needle);
      } else {
         k = strstr (k, needle);
      }
      if (k) {                                     // if the needle is found then perform a word test
         r = test_for_word (haystack, needle, k);
         if (r) break;                             // break if the word test succeeds
         k++;                                      // advance one character looking for other occurances
      }
   }
   return r;                                       // return the final result
}

// search the file per the specified criteria and settings
void search (FILE *f, char *needle, bool word_search, bool invert_search, bool ignore_case) {
   char haystack[256];
   int r;
   bool b;
   while (dylib.fgets (haystack, sizeof (haystack), f)) { // loop through the entire file
      if (word_search) {                                  // if a word search then so do
         b = has_word (haystack, needle, ignore_case);
      }  else {
         // test the input line for the search string, accounting for case as specified
         if (ignore_case) {
            b = strcasestr (haystack, needle) != NULL;
         } else {
            b = strstr (haystack, needle) != NULL;
         }
      }
      if (invert_search) b = !b;                          // invert the search if requested
      if (b) {
         // criteria matched, so output this line
         r = dylib.fputs (haystack, stdout);
         assert (r >= 0);
      }
   }
}

// display usage
void display_usage () {
   dylib.fputs ("usage: grep [-ivw] pattern [file ...]\n", stderr);
}

// the main grep method
int main (int argc, char *argv[]) {

   // capture the options
   int opt;
   bool invert_search = false;
   bool word_search   = false;
   bool ignore_case   = false;
   while ((opt = getopt (argc, argv, "viw")) != -1) {
      switch (opt) {
      case 'v':
         invert_search = true;
         break;
      case 'i':
         ignore_case = true;
         break;
      case 'w':
         word_search = true;
      case '?':
      default:
         break;
      }
   }

   // adjust argc and argv removing the processed options
   argc -= optind;
   argv += optind;

   // now that arguments have been processed, if present,
   // ensure there remains a search string, any additional args are files
   if (argc >= 1) {
      // capture the search string
      char *ss = argv[0];
      if (argc == 1) {
         search (stdin, ss, word_search, invert_search, ignore_case);
      } else {
         FILE *f;
         for (int i = 1; i < argc; i++) {
            f = dylib.fopen (argv[i], "r");
            if (f) {
               search (f, ss, word_search, invert_search, ignore_case);
               dylib.fclose (f);
            } else {
               dylib.fputs ("grep: ", stderr);
               dylib.fputs (argv[i], stderr);
               dylib.fputs (": No such file or directory\n", stderr);
            }
         }
      }
   } else {
      display_usage ();
   }

   return 0;
}
