/* head.c
   This program display first lines of a file.
   Usage: head [-n count] [-q] [file ...]
   Options:
   -n number of lines
   -q suppress printing of filename headers when multiple files are printed
   change log:
   08/18/2023 initial version
   08/20/2023 added printing of filename headers when multiple files are printed
   08/20/2023 added -q option
   08/20/2023 changed use of fprintf for dylib.fputs to reduce memory footprint
   12/10/2023 removed reference to string_ext.h
   02/27/2024 updated to use dylib
   01/20/2025 moved fputs to dylib
   06/08/2025 updated fgets / strcspn
   06/16/2025 updated exit reference
*/

#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <stdbool.h>
#include <stdio.h>
#include <dylib.h>

// prints usage info
void print_usage () {
   dylib.fputs ("usage: head [-n lines] [file ...]\n", stdout);
}

// processes one file
void process_file (FILE *f, int max_count, bool print_filename, const char *filename) {
   static int fc = 0;                        // file counter
   char s[256];                              // input buffer
   int n = 1;                                // line counter

   if (print_filename) {                     // print headers
      if (fc) {                              // print second and subsequent file breaks
         dylib.fputs ("\n", stdout);
      }
      fc++;                                  // increment the file counter
      dylib.fputs ("==> ", stdout);          // print the file header
      dylib.fputs (filename, stdout);
      dylib.fputs (" <==\n", stdout);
   }
      
   while (dylib.fgets (s, sizeof (s), f)) {  // read from the file
      dylib.fputs (s, stdout);               // output the text
      if (n >= max_count) {
         break;
      }
      n++;
   }
}

// the head program
int main (int argc, char *argv[]) {

   int max_count              = 10;
   bool error                 = false;
   bool pr_headers_mult_files = true;

   // capture the options
   int opt;
   while ((opt = getopt (argc, argv, "n:q")) != -1) {
      switch (opt) {                            
         case 'n':                             
            max_count = atoi (optarg);
            if (max_count <= 0) {
               dylib.fputs ("head: illegal line count\n", stderr);
               error = true;
            }
            break; 
         case 'q':
            pr_headers_mult_files = false;
            break;
         case '?':
         default:
            error = true;
            break;
      }
   }

   // exit if there was an error while processing options
   if (error) {
      print_usage ();
      exit (0);
   }

   // adjust argc and argv removing the processed options
   argc -= optind; 
   argv += optind; 

   if (argc) {
      // loop through all files
      for (int i = 0; i < argc; i++) {
         // open the file
         FILE *f = dylib.fopen (argv[i], "r");
         // confirm the file is open
         if (f) {
            process_file (f, max_count, pr_headers_mult_files && (argc > 1), argv[i]);
            dylib.fclose (f);
         } else {
            // file failed to open
            dylib.fputs ("head: ", stderr);
            dylib.fputs (argv[i], stderr);
            dylib.fputs (": no such file or directory\n", stderr);
         }
      }
   } else {
      process_file (stdin, max_count, false, NULL);
   }

   return 0;
}
