/* hello.c
   This program provides a simple hello world implementation to demonstrate
   the simplest possible program that does something.
   change log:
   07/27/2023 initial version
   08/14/2023 simplified code down to the "standard" hello world in proper C form.
   01/04/2025 removed test code
*/

#include <stdio.h>
#include <stdbool.h>
#include <string.h>
#include <stdlib.h>
#include <dylib.h>

#define BASIC_FILES_MAX          3
#define BASIC_FILES_TEXT_LEN_MAX 128

typedef struct {
   bool in_use;
   int id;
   FILE *f;
   char data[BASIC_FILES_TEXT_LEN_MAX];
} basic_file_t;

basic_file_t basic_file[BASIC_FILES_MAX];

void basic_files_init () {
   memset (&basic_file, 0x00, sizeof (basic_file));
}

const char *basic_file_mode[3] = {"r", "w", "a"};

void basic_file_read_data (int index) {
   dylib.strcpy (basic_file[index].data, "");
   dylib.fgets (basic_file[index].data, BASIC_FILES_TEXT_LEN_MAX, basic_file[index].f);
   basic_file[index].data[strcspn (basic_file[index].data, "!\r\n")] = 0x00;  // kill of CR and LF
}

void basic_file_open (int id, const char *name, int mode) {
   int r = 1;
   for (int i = 0; i < BASIC_FILES_MAX; i++) {
      if (!basic_file[i].in_use) {
         basic_file[i].f      = dylib.fopen (name, basic_file_mode[mode]);
         if (basic_file[i].f) {
            basic_file[i].id     = id;
            basic_file[i].in_use = true;
            if (!mode) {   // if the file is being opened for reading, read the first line so EOF will be triggered ahead of
                           // the basic program input call
               basic_file_read_data (i);
            }
            r = 0;
         }
         break;
      }
   }
   if (r) {
      fprintf (stdout, "cannot open file %s id %d\n", name, id);
      exit (0);
   }
}

int basic_file_get_index (int id) {
   int r = -1;
   for (int i = 0; i < BASIC_FILES_MAX; i++) {
      if (basic_file[i].in_use && basic_file[i].id == id) {
         r = i;
         break;
      }
   }
   if (r < 0) {
      fprintf (stdout, "file id %d not open\n", id);
      exit (0);
   }
   return r;
}

void basic_file_close (int id) {
   int i = basic_file_get_index (id);
   fclose (basic_file[i].f);
   basic_file[i].in_use = false;
}

void basic_file_print (int id, const char *s) {
   int i = basic_file_get_index (id);
   fprintf (basic_file[i].f, "%s\n", s);
}

void basic_file_input (int id, char *s) {
   int i = basic_file_get_index (id);
   dylib.strcpy (s, basic_file[i].data);
   basic_file_read_data (i);
}

int eof (int id) {
   int r = 1;
   int i = basic_file_get_index (id);
   r = basic_file[i].f->eof;
   return r;
}

int main (int argc, char *argv[]) {

   fputs ("Hello World!\n", stdout);

   basic_files_init ();
   basic_file_open (64, "btestf", 1);
   basic_file_print (64, "basic test, line 1 (more)");
   basic_file_print (64, "basic test, line 2 (more)");
   basic_file_print (64, "basic test, line 3 (last)");
   basic_file_close (64);

   char s[128];
   basic_file_open (6, "btestf", 0);
   while (!eof (6)) {
      basic_file_input (6, s);
      fprintf (stdout, "%s\n", s);
   }
   basic_file_close (6);

   return 0;
}
