/* hexdump.c
   This program implements the hexdump command.
   to do
   change history
   06/06/2025 initial version
   06/07/2025 added display of diagnostic messages. These are problematic as, although written to stderr, are interspersed within the output, and must
              not upset that output, waiting until an appropriate time to output on a separate line. Goodness who thought to do that? Should have scanned
              the input files first and printed diagnostic messages before output of processed data.
              Moved from use of ptoa (upper case) to ptoa_lower.
   06/08/2025 updated to output only full lines at a time, regardless of regular output or diagnostic messages
              updated to output total bytes
   06/09/2025 output cleanup
   06/16/2025 updated exit reference
   07/05/2025 dylib adjustments
   09/06/2025 corrected printable character printing to exclude the value 128
*/

#include <unistd.h>
#include <stdbool.h>
#include <stdio.h>
#include <sys/stat.h>
#include <string.h>
#include <conversion.h>
#include <dylib.h>
#include <stdlib.h>

#define DISPLAY_MODE_WORDS       0       // display mode showing words
#define DISPLAY_MODE_ASCII_CHARS 1       // display mode showing individual characters and a textual representation

char error_messages[256]   = {0x00};     // error messages
bool first_char            = true;       // value inidicating whether this is the first or second character of a word
unsigned long int file_pos = 0;          // file position
char line[128]             = {0x00};     // line of output text
char ascii_display[20]     = {0x00};     // line of displayable ascii text

void print_error_messages () {           // print error messges
   dylib.fputs (error_messages, stderr);
   dylib.strcpy (error_messages, "");
}

void clear_ascii_display () {            // clears the ascii displayable text
   memset (ascii_display, 0x00, sizeof (ascii_display));
}

// handle processing of ascii chars and visible text
void display_ascii_chars (unsigned char *s, int n) {

   unsigned long int rem;

   if (n) {
      for (int i = 0; i < n; i++) {

         rem = file_pos % 16;

         // print line position
         if (!rem) {

            dylib.strcpy (line, "");

            print_error_messages ();

            dylib.strcat (line, ptoa_lower (&file_pos, sizeof (unsigned long int)));
            dylib.strcat (line, " ");
         }
   
         if (rem == 8) {
            dylib.strcat (line, " ");
         }
    
         // print character
         dylib.strcat (line, ptoa_lower (&s[i], sizeof (unsigned char)));
         dylib.strcat (line, " ");
   
         if (s[i] >= 32 && s[i] < 128) {
            ascii_display[rem] = s[i];
         } else {
            ascii_display[rem] = '.';
         }
   
         if (rem == 15) {
            dylib.strcat (line, " |");
            dylib.strcat (line, ascii_display);
            clear_ascii_display ();
            dylib.strcat (line, "|");
            dylib.strcat (line, "\n");
            dylib.fputs (line, stdout);
            dylib.strcpy (line, "");
         }

         file_pos++;
      }
   } else {
      unsigned long total_bytes = file_pos;

      if (file_pos % 16) {
         while (1) {
            rem = file_pos % 16;

            if (rem == 8) {
               dylib.strcat (line, " ");
            }
   
            // print character
            dylib.strcat (line, "   ");
   
            if (rem == 15) {
               dylib.strcat (line, " |");
               dylib.strcat (line, ascii_display);
               clear_ascii_display ();
               dylib.strcat (line, "|");
               dylib.strcat (line, "\n");
               dylib.fputs (line, stdout);
               dylib.strcpy (line, "");
   
               break;
            }

            file_pos++;
         }
      }

      if (file_pos) {
         dylib.strcat (line, ptoa_lower (&total_bytes, sizeof (unsigned long int)));
         dylib.strcat (line, "\n");
         dylib.fputs (line, stdout);
         dylib.strcpy (line, "");
      }

      print_error_messages ();
   }
}

// handle display of words
void display_words (unsigned char *s, int n) {

   if (n) {
      for (int i = 0; i < n; i++) {

         // print line position
         if (file_pos % 16 == 0) {
   
            dylib.strcpy (line, "");

            print_error_messages ();
   
            dylib.strcat (line, ptoa_lower (&file_pos, sizeof (unsigned long int)));
            dylib.strcat (line, " ");
         }
   
         // print character
         dylib.strcat (line, ptoa_lower (&s[i], sizeof (unsigned char)));
         if (first_char) {
            first_char = false;
         } else {
            dylib.strcat (line, " ");
            first_char = true;
         }

         if (file_pos % 16 == 15) {
            dylib.strcat (line, "\n");
            dylib.fputs (line, stdout);
            dylib.strcpy (line, "");
         }

         file_pos++;
      }        
   } else {
      if (file_pos % 16) {
         dylib.strcat (line, "\n");
         dylib.fputs (line, stdout);
         dylib.strcpy (line, "");
      } else {
         dylib.fputs ("\n", stdout);
      }

      if (file_pos) {
         dylib.fputs (ptoa_lower (&file_pos, sizeof (unsigned long int)), stdout);
         dylib.fputs ("\n", stdout);
      }

      print_error_messages ();
   }
}

// process read data to the appropriate display mode
void process_data (unsigned char *s, int n, int display_mode) {
   if (display_mode == DISPLAY_MODE_WORDS) {
      display_words (s, n);
   } else if (display_mode == DISPLAY_MODE_ASCII_CHARS) {
      display_ascii_chars (s, n);
   }
}

// process incoming files with the appropriate display mode
int process_file (const char *path, int display_mode) {
   int r = 0;

   // read and write modes by TI file_type. Blank strings are for types currently unsupported.
   // stat values for text files are 2, and binary files 3
   const char *read_mode[]  = {"", "", "r", "rb", "", ""};

   unsigned char s[255];
   int n;

   struct stat statbuf;
   r = stat (path, &statbuf);
   if (!r) {

      // open the file
      FILE *f = dylib.fopen (path, read_mode[statbuf.st_mode]);
      if (f) {
   
         // propcess the file contents
         if (statbuf.st_mode == 3) {
            while ((n = dylib.fread (s, 1, sizeof (s), f))) {
               process_data (s, n, display_mode);
            }
         } else {
            while (dylib.fgets ((char *) s, sizeof (s), f)) {
               process_data (s, dylib.strlen ((char *) s), display_mode);
            }
         }
   
         // close the source
         dylib.fclose (f);

      } else {
         dylib.strcat (error_messages, "hexdump: ");
         dylib.strcat (error_messages, path);
         dylib.strcat (error_messages, ": Is a directory\n");

         r = 2; // not a file -- is a directory
      }
   } else {
      dylib.strcat (error_messages, "hexdump: ");
      dylib.strcat (error_messages, path);
      dylib.strcat (error_messages, ": No such file or directory\n");

      r = 1; // doesn't exist
   }

   return r;
}

// process stdin
void process_stdin (int display_mode) {
   unsigned char s[255];
   while (dylib.fgets ((char *) s, sizeof (s), stdin)) {
      process_data (s, dylib.strlen ((char *) s), display_mode);
   }
}

// displays program usage
void display_usage () {
   dylib.fputs ("usage: hexdump [-C] [file ...]\n", stdout);
}

// the main procedure
int main (int argc, char *argv[]) {

   int display_mode = DISPLAY_MODE_WORDS;

   // capture the options
   int opt;
   int error = false;
   while ((opt = getopt (argc, argv, "C")) != -1) {
      switch (opt) {
         case 'C':                                           
            display_mode = DISPLAY_MODE_ASCII_CHARS;
            break;
         case '?':                                          
         default:
            error = true;       
            break;
      }
   }

   if (error) {
      display_usage ();
      exit (1);
   }

   // adjust argc and argv removing the processed options
   argc -= optind;
   argv += optind;

   if (argc) {
      for (int i = 0; i < argc; i++) {
         process_file (argv[i], display_mode);
      }
      process_data (NULL, 0, display_mode); // conclude any output lines still in progress, including any remaining diagnostic messages
   } else {
      process_stdin (display_mode);
      process_data (NULL, 0, display_mode);
   }

   return 0;
}
