/* imagedraw.c
   This program draws an image at the the top left of the screen or optionally overridden origin.
   Usage: bitmap_3
   change log:
   01/27/2025 initial version
   02/28/2025 updated for removal of 80x30 display mode
   06/05/2025 signal handling
   06/07/2025 moved calls to dylib
   07/08/2025 updated to use the image path as the cache name
*/

#include <console.h>
#include <dylib.h>
#include <unistd.h>
#include <stdlib.h>

// prints usage info
void print_usage () {
   dylib.fputs ("usage: imagedraw [-y yorg] [-x xorg]  file\n", stderr);
} 

int recover_display_mode;
int recover_display_rows;

// restore settings before exiting
void restore_screen () {
   console_display_set_mode (recover_display_mode, recover_display_rows);
   console_cls ();
   console_standard_set_default_color ();
   console_fonts_load_std ();
}

int main (int argc, char *argv[]) {

   int y = 0;
   int x = 0;

   // capture the options
   int opt;
   bool error        = false;

   while ((opt = getopt (argc, argv, "x:y:")) != -1) {
      switch (opt) {
      case 'x':
         x = atoi (optarg); 
         break;
      case 'y':
         y = atoi (optarg); 
         break;
      case '?':
         error = true;
         break;
      default:
         break;
      }
   }

   // adjust argc and argv removing the processed options
   argc -= optind;
   argv += optind;

   if (!error && argc == 1) {

      const char *image_path = argv[0];

      // save the current display mode
      console_display_get_mode (&recover_display_mode, &recover_display_rows);

      // load the image while we have file access
      int bytes = console_bitmap_load_image (image_path, image_path);

      if (bytes > 0) {

         files (0);

         // switch to bitmap mode
         console_display_set_mode (DISPLAY_MODE_BITMAP, DISPLAY_ROWS_24);
         
         // since the screen mode changed set the atexit to restore the screen
         dylib.atexit (restore_screen);
  
         // clear the screen
         console_bitmap_cls (CONSOLE_BITMAP_IMAGE_TABLE_FILL_INCREMENTING);
   
         // set the patterns
         unsigned char char_pattern[] = {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00};
         console_bitmap_set_all_patterns (char_pattern);
   
         // set the colors
         unsigned char color_pattern[] = {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00};
         console_bitmap_set_all_colors (color_pattern);
   
         // demonstrate raw bitmap mode
         console_bitmap_draw_image (image_path, y, x);

         console_getc ();
 
         files (3);

      } else {
         dylib.fputs ("imagedraw: ", stderr);  
         dylib.fputs (image_path, stderr);   
         dylib.fputs (": cannot load file\n", stderr);
      }
   } else {
       error = true;
   }

   if (error) {
       print_usage ();
   }

   return 0;
}
