/* console.h
   This header defines the console I/O package, which provides text, graphics, keyboard
   and joystick support with the console, based on libti99. 
   It relies on libti99's vdp package only.
   change log
   07/01/2023 initial version
   07/21/2023 added DISPLAY_MODE_F18A_TEXT80
   08/18/2023 changed the default console name to a con0
   09/10/2023 added sprite methods
   10/21/2023 removed all window references
   11/01/2023 added console_text_get_foreground_color, console_text_get_background_color
              added console_display_get_mode
   11/08/2023 added secure mode to console_gets
   02/24/2024 broad support for shared libraries
   12/11/2024 updated console_font_get_text to use char rather than unsigned char
   12/26/2024 updated header comments for accuracy
   12/28/2024 updated default colors to dark green on black
   01/11/2025 renamed sprite magnification method
              added bitmap methods
   01/13/2025 removed console_bitmap_clear_all_patterns
              continued sorting to determine best bitmap methods
   01/20/2025 renamed con0 /dev/console
   01/25/2025 updated parameters for bitmap image load and draw
   02/02/2025 added F18A_TEXT80x30
   02/28/2025 added DISPLAY_ROWS_*
              added rows parameter to console_display_set_mode
              modified console_display_get_mode to return mode and rows
              removed DISPLAY_MODE_F18A_TEXT80x30
   06/12/2025 added sprite coincidence parameter
   10/21/2025 added console_display_flash
*/

#ifndef CONSOLE_H
#define CONSOLE_H

#include <stdarg.h>
#include <stdbool.h>

#define JOYSTICK_LEFT        -1
#define JOYSTICK_RIGHT       1
#define JOYSTICK_UP          -1
#define JOYSTICK_DOWN        1

#define CONSOLE_WINDOW_DEFAULT "/dev/console"

// DISPLAY MODE

#define DISPLAY_MODE_STANDARD       0
#define DISPLAY_MODE_TEXT           1
#define DISPLAY_MODE_MULTICOLOR     2
#define DISPLAY_MODE_BITMAP         3
#define DISPLAY_MODE_F18A_TEXT80    4

#define DISPLAY_ROWS_24      24
#define DISPLAY_ROWS_30      30
#define DISPLAY_ROWS_DEFAULT DISPLAY_ROWS_24

// returns the console display mode
void console_display_get_mode (int *mode, int *rows);

// sets the console display mode
void console_display_set_mode (int mode, int rows);

// returns the number of rows on the display
int console_display_get_rows ();

// returns the number of columns on the display
int console_display_get_columns ();

// flashes the console display
void console_display_flash (); 

// TEXT I/O

// clears the window
void console_cls ();

// puts a string in the window with control character support
int console_puts (const char *s);

// writes a string in the window at the specified location with no control character support
void console_write_raw (int y, int x, const char *s, int len);

// reads a character in the window at the specified location
int console_read_raw_char (int y, int x);

// puts a character in the window
int console_putc (int c);

// gets a string from the window
char *console_gets (char *buf, int maxlen, bool secure);

// gets a character, waiting until key press
int console_getc ();

// returns the current key pressed, if any
int console_get_key ();

// returns the current joystick state
void console_joystick_read (int jdev, int *y, int *x);

// sets the text writing position in the window
void console_set_pos (int x, int y);

// gets the text writing position in the window
void console_get_pos (int *x, int *y);

// returns the window size
void console_get_window_size (int *rows, int *cols);

// TEXT COLOR

#define COLOR_TRANS    0x00
#define COLOR_BLACK    0x01
#define COLOR_MEDGREEN 0x02
#define COLOR_LTGREEN  0x03
#define COLOR_DKBLUE   0x04
#define COLOR_LTBLUE   0x05
#define COLOR_DKRED    0x06
#define COLOR_CYAN     0x07                                            
#define COLOR_MEDRED   0x08
#define COLOR_LTRED    0x09
#define COLOR_DKYELLOW 0x0A
#define COLOR_LTYELLOW 0x0B
#define COLOR_DKGREEN  0x0C
#define COLOR_MAGENTA  0x0D
#define COLOR_GRAY     0x0E
#define COLOR_WHITE    0x0F

// default text colors

#define COLOR_DEFAULT_TEXT_FOREGROUND COLOR_DKGREEN
#define COLOR_DEFAULT_TEXT_BACKGROUND COLOR_BLACK

// sets the console border color
int console_border_color_set (unsigned int x);

// gets the console border color
int console_border_color_set ();

// sets the default text foreground color (text mode)
void console_text_set_default_color ();

// gets the text background color
unsigned int console_text_get_background_color ();

// sets the text background color (text mode)
int console_text_set_background_color (unsigned int color);

// gets the text foreground color
unsigned int console_text_get_foreground_color ();

// sets the text foreground color (text mode)
int console_text_set_foreground_color (unsigned int color);

// sets the default text colors (standard mode)
void console_standard_set_default_color ();

// sets the color of a character group
int console_standard_set_char_group_color (int g, int fg_color, int bg_color);

// sets the color of all character groups
int console_standard_set_all_colors (int fg_color, int bg_color);

// SPRITES

// sets the console sprite mode
int console_sprite_set_magnification (int sprite_mode);

// sets the console sprite values
void console_sprite_set_values (int sprite_index, int chardef, int color, int pos_y, int pos_x, int vel_y, int vel_x);

// sets the console sprite character
void console_sprite_set_char (int sprite_index, int chardef);

// sets the console sprite pattern. Note that in standard and text_80 modes the sprite and text character patterns overlap.
int console_sprite_set_pattern (int c, const unsigned char *pattern);

// sets the console sprite color
void console_sprite_set_color (int sprite_index, int color);

// gets the console sprite position
void console_sprite_get_position (int sprite_index, int *y, int *x);

// sets the console sprite position
void console_sprite_set_position (int sprite_index, int y, int x);

// set the console sprite motion
void console_sprite_set_motion (int sprite_index, int vel_y, int vel_x);

// enables console sprite motion 
void console_sprite_enable_motion ();

// disables console sprite motion
void console_sprite_disable_motion ();

// reports a coincidence between two sprites. sensitivity generally is the width of the sprite to test for coincidence. 7 will test for full coverage of the sprite
// block, whereas 1 will provide a lot of grace
bool console_sprite_coincidence (int sprite_index_1, int sprite_index_2, int sensitivity);

// reports coincidence with a target sprite with a range of other sprites
bool console_sprite_coincidence_target (int sprite_target_index, int sprite_source_start_index, int sprite_source_end_index, int sensitivity);

// tests a sprite's position for alignment with the character display
void console_sprite_test_char_alignment (int sprite_id, bool *is_aligned, int *y, int *x);

// retries the sprite's position and velocity
void console_sprite_get_values (int sprite_index, int *pos_y, int *pos_x, int *vel_y, int *vel_x);

// FONTS

// returns the font width
int console_font_get_width ();

// loads the standard fonts
void console_fonts_load_std ();

// gets the font for a character in raw font format (8 total characters), in graphics i and text mode
int console_font_get (int c, unsigned char *font);

// gets the font for a character in the textual format, in graphics i and text mode
int console_font_get_text (int c, char *text_font);

// sets the font for a character with the raw font (8 total characters), in graphics i and text mode
int console_font_set (int c, const unsigned char *font);

// sets the font for a character with the textual font to be interpretted (16 total characters), in graphics i and text mode
int console_font_set_text (int c, const char *text_font);

// writes all character fonts to the specified file, returns 0 on success, in graphics i and text mode
int console_font_dump (const char *filename);

// loads character patters form the specified file, returns 0 on success, in graphics i and text mode
int console_font_load (const char *filename);

// SCREEN BLANKING

// resets the blanking timeout
void console_reset_blanking ();

#define CONSOLE_BITMAP_IMAGE_TABLE_FILL_ZERO         0
#define CONSOLE_BITMAP_IMAGE_TABLE_FILL_INCREMENTING 1
#define CONSOLE_BITMAP_REGIONS 3

// CONFIRMED OF VALUE

void console_bitmap_cls (int fill_pattern);

void console_bitmap_set_all_patterns (unsigned char *pattern);
void console_bitmap_set_pattern2 (int y, int x, const unsigned char *s, int len);

void console_bitmap_set_all_colors (unsigned char *pattern);
void console_bitmap_set_color_pattern2 (int y, int x, const unsigned char *s, int len);

void console_bitmap_draw_pixel2 (int y, int x);
void console_bitmap_clear_pixel (int y, int x);

void console_bitmap_draw_line_right (int y, int x, int len);
void console_bitmap_clear_line_right (int y, int x, int len);

void console_bitmap_draw_line_down (int y, int x, int len);
void console_bitmap_clear_line_down (int y, int x, int len);

void console_bitmap_draw_line2 (int y0, int x0, int y1, int x1);
void console_bitmap_clear_line (int y0, int x0, int y1, int x1);

// writes text in bitmap mode at the character position (y, x)
void console_bitmap_write_text_raw (int y, int x, const char *s);

int console_bitmap_load_image (const char *cache_name, const char *path);
void console_bitmap_draw_image (const char *cache_name, int y, int x);

// MAYBE OF VALUE

void console_bitmap_set_char (int region, int y, int x, int id);
void console_bitmap_set_pattern (int region, int id, unsigned char *pattern);
void console_bitmap_set_color_pattern (int region, int id, unsigned char *pattern);

//TBD

#define CONSOLE_BITMAP_CLEAR 0
#define CONSOLE_BITMAP_SET   1

void console_bitmap_draw_pixel (int y, int x, int mode);

void console_bitmap_set_pixel_color (int y, int x, unsigned char color);

void console_bitmap_set_pixel_and_color (int y, int x, int mode, unsigned char color);

void console_bitmap_draw_line (int y0, int x0, int y1, int x1, int mode);

#endif
