/* gpu.h
   This header defines the types and methods for use of the gpu methods api.

   UNIX99 uses the F18A GPU, when available, for scrolling the screen and clearing the screen. Other functions will be added
   as needed.

   Applications may also use the F18A GPU. The rules, although straight forward, are also somewhat limiting:
   - Methods must fit in the available space, which is constrained by the display mode. All modes except bitmap have about 6KB,
     as space is reserved for full features of the F18A. Bitmap mode has only around 2 KB available.
   - Methods must be written with exclusively relocatable (relative) code. Calls to direct addresses are likely to fail.
   - Space is provided for method parameterization and results in the first 0x20 bytes of the gpu method region. For all 
     display modes except bitmap, this range is 0x2000-0x201f. 0x2000-0x2005 are used for UNIX99 parameters and return results:
     0x2000 gpu_exists (TBD--may be used by apps as the result is cached in CPU RAM)
     0x2002 display rows
     0x2004 display columns
   - Similarly space is provided in the bitmap mode at 0x1800 for 0x20 bytes. Layout is TBD.

   The method gpu_get_method_addr can be used to determine it's load address if the app developer must write non-relocatable code.
   This is obviously highly discouraged since the size of other GPU methods may change, resulting in an address offset.

   gpu_load_method is used to load a gpu method into vdp ram and prepare it for use. the method id must be selected from the 
   GPU_APP_METHOD_* values.

   gpu_method_purge can be used to load new methods when the previously loaded set are no longer needed. The IDs and allocated
   space are cleared. This method is called by _start at the beginning of every program (before main). Note that UNIX99 does
   retain GPU programs it loads at startup and during every call to console_display_set_mode. If retention of GPU methods is
   desirable, it's a fairly easy API add, but one that must be addressed gingerly.

   change history
   02/14/2025 initial version
   02/15/2025 added detailed description to this header
   02/19/2025 added gpu_is_present
*/

#ifndef GPU_H
#define GPU_H

#define GPU_APP_METHOD_0         7
#define GPU_APP_METHOD_1         8
#define GPU_APP_METHOD_2         9
#define GPU_APP_METHOD_3        10
#define GPU_APP_METHOD_4        11
#define GPU_APP_METHOD_5        12
#define GPU_APP_METHOD_6        13
#define GPU_APP_METHOD_7        14
#define GPU_APP_METHOD_8        15

// returns true if a gpu is present
int gpu_is_present ();

// loads a GPU method into VRAM; return 0 success, non-zero failure
int gpu_load_method (int method_id, unsigned char *cpu_method_addr, int size);

// executes a GPU method. return is 0 success, non-zero failure
int gpu_execute (int method_id);

// returns the gpu method address in vdp ram
unsigned int gpu_get_method_addr (int method_id);

// purges all application-specific methods from VDP RAM and makes the system ready to load new application-specific methods
void gpu_method_purge ();

#endif
