/* basic.h
   This header defines the BASIC language types and methods used by the BASIC to C translator.
   change log
   11/22/2025 initial version
*/

#ifndef BASIC_H
#define BASIC_H

#include <stdbool.h>
#include <stdio.h>

#define CSTRING_LEN 128

// approximation of BASIC string type in C
typedef struct {
   char val[CSTRING_LEN];
} string_t;

extern double pi;

// convert a string literal to a string_t
string_t string_assign (const char *s);

// prints a string_t
void string_print (FILE *stream, string_t s);

// reads a value into a string_t
void string_input (string_t *s);

// concatenates two strings and returns the result
string_t concat (string_t a, string_t b);

// compares two strings and returns the result: -1 a < b, 0 a = b, 1 a > b
int compare (string_t a, string_t b);

// returns the leftmost count characters in a string
string_t left (string_t a, int b);

// return the length of a string
int len (string_t a);

// return a string segment
string_t seg (string_t src, int start, int length);

// returns the right most characters of a string
string_t right (string_t src, int length);

// returns the position of a sub string in a source string, starting at start
int pos (string_t src, string_t sub, int start);

// returns the ascii value of the first character of a string
int asc (string_t s);

// returns the string representing the ascii value
string_t chr (int n);

// returns the basic version
double version ();

// clears the screen
void clear ();

// sets the display mode
void display_mode (int mode, int rows);

void display_mode_get (int *mode);
void display_rows_and_cols (int *rows, int *cols);

// sets the colors of a color group in VDP graphics I mode
void color (int g, int fc, int bc);

// generates a repeated string
string_t rpt (string_t src, int count);

// sets the random number seed
void randomize (unsigned r);

// returns a random number between 0 and < 1
double rnd ();

// returns a random number between 0 and x - 1
int rndi (int x);

// sets sprite values
void sprite (int sprite_index, int chardef, int color, int pos_y, int pos_x, int vel_y, int vel_x);

void coinc (int sprite_index1, int sprite_index2, int tolerance, int *result);

// get sprite position
void position (int sprite_index, int *y, int *x);

// locates a sprite
void locate (int sprite_index, int y, int x);

// sets a character definition
void CHAR (int c, string_t pattern);

// returns the current key pressed
int inkey ();

// writes a repeated character across the display horizontally or vertically from a start position
void hchar (int y, int x, int c, int count);
void vchar (int y, int x, int c, int count);

void gchar (int y, int x, int *c);

// sets the cursor position
void at (int y, int x);
void atend ();

#define BASIC_FILES_MAX 3

typedef struct {
   bool in_use;
   int id;
   FILE *f;
   string_t s;
} basic_file_t;

extern basic_file_t basic_file[BASIC_FILES_MAX];

void basic_files_init ();

void basic_file_read_data (int index);

void basic_file_open (int id, string_t name, int mode);

int basic_file_get_index (int id);

FILE *basic_file_get_stream (int id);

void basic_file_close (int id);

void basic_file_input (int id, string_t *s);

int eof (int id);

int vali (string_t s);
long vall (string_t s);
double vald (string_t s);

string_t stri (int v);
string_t strl (long v);
string_t strd (double v);

void joyst (int id, int *y, int *x);

#endif

