/* console_private.h
   This header defines the data, types and methods required for private console functionality.
   change log:
   07/01/2023 initial version
   10/20/2023 migrated extern vdp and console values to the cache so the values persist from program to program
   10/21/2023 removed all window references
   10/21/2023 removed fonts_are_loaded from state
   11/04/2023 added motion enabled flag for sprites
   02/14/2024 added SIT_REG_FRAME_*, SIT_ADDR_FRAME_* and sit_frame_index to support frame swapping for "fast" screen scroll
   02/16/2024 modified def for console_scroll_one to return an integer
   02/24/2024 broad changes for introduction of the dylib package
   01/11/2025 added definitions for vdp reg1 for each display mode
   01/18/2025 added console_font_load_for_kernel 
   01/22/2025 performance improvements (console state data in 16-bit scratch ram)
   01/24/2025 removed row, col in console_state_t
   01/26/2025 updated parameters for console_bitmap_load and draw
   02/02/2025 added F18 text 80x30 register values
   02/11/2025 removed SIT register and address values
   02/13/2025 added void kernel_console_display_set_mode_legacy_parameters
   02/28/2025 removed DISPLAY_MODE_F18A_TEXT80x30_REG1
   04/30/2025 added console_display_unblank
   08/25/2025 added kernel_console_disable_quit_key
   10/23/2025 added kernel_console_display_flash
*/

#ifndef CONSOLE_PRIVATE_H
#define CONSOLE_PRIVATE_H

#include <stdbool.h>

typedef struct {
   int mode;
   unsigned int vdp_screen_color;
   unsigned int vdp_screen_image_table;      // SIT, Register 2 * 0x400
   unsigned int vdp_color_table;             // CR,  Register 3 * 0x40
   unsigned int vdp_character_pattern_table; // PDT, Register 4 * 0x800
   unsigned int vdp_sprite_attribute_table;  // SAL, Register 5 * 0x80
   unsigned int vdp_sprite_pattern_table;    // SDT, Register 6 * 0x800
   int screen_height;
   int screen_width;
   unsigned int vdp_text_addr_current;
   unsigned int vdp_text_addr_scroll_reqd;
} console_state_t;

#define MAX_NUM_SPRITES 32

#define SPRITE_ROW_MIN   0
#define SPRITE_ROW_MAX 190
#define SPRITE_COL_MIN   0
#define SPRITE_COL_MAX 255

typedef struct {
   int chardef;
   int color;
   int pos_y;
   int pos_x;
   int vel_y;
   int vel_x;
} sprite_t;

typedef struct {
   bool motion_enabled;
   int last_used_index;
   int last_motion_index;
   sprite_t sprite[MAX_NUM_SPRITES];
} sprite_info_t;

#define DISPLAY_MODE_BITMAP_REG0          0x02

#define DISPLAY_MODE_STANDARD_VDP_REG1    (VDP_MODE1_16K | VDP_MODE1_UNBLANK)
#define DISPLAY_MODE_TEXT_VDP_REG1        (VDP_MODE1_16K | VDP_MODE1_TEXT | VDP_MODE1_UNBLANK)
#define DISPLAY_MODE_F18A_TEXT80_REG1     (VDP_MODE1_16K | VDP_MODE1_TEXT | VDP_MODE1_UNBLANK)
#define DISPLAY_MODE_BITMAP_REG1          (VDP_MODE1_16K | VDP_MODE1_UNBLANK)

extern sprite_info_t sprite_info;

void console_init ();

void console_text_draw_color ();

void console_cache_init ();

void console_update ();

void console_sprite_update ();

void console_sprite_init ();

void console_sprite_init_motion ();

// sets sprite vdp values
void console_sprite_set_vdp_vals (int n, int ch, int col, int r, int c);

void console_prepare_bottom_row ();
int console_scroll_one ();

void console_display_set_mode_legacy_parameters ();

void console_display_unblank ();

void kernel_console_font_load ();

void kernel_console_display_flash ();

#define SAMS_ADDR_PATTERN 0xa000
#define SAMS_ADDR_COLOR (SAMS_ADDR_PATTERN + 0x1800)
#define VDP_TABLE_LEN 0x1800
int kernel_console_bitmap_load_image (const char *cache_name, const char *path);
void kernel_console_bitmap_draw_image (const char *cache_name, int y, int x);

void kernel_console_disable_quit_key ();

#endif
