/* dylib_private.h
   This header defines external definitions for the dynamic library.
   change log:
   02/18/2024 initial version
   02/24/2024 broad support for shared libraries
   02/25/2024 added stdlib, stdio, sequencer methods
   02/29/2024 added generate_unix_path
   03/01/2024 added multiple methods
   03/07/2024 added spinner methods
              added getcwd
   12/27/2024 removed file_atexit
   01/07/2025 added reboot
   01/10/2025 added fwrite, fseek, ftell, fputs, file_get_available
   01/17/2025 added file_atexit
   01/18/2025 removed dir_init and file_init
   01/20/2025 added console_puts, file_open_std_files, console_write_raw
              removed fopen_pio, fopen_console
   01/22/2025 performance improvements
   01/27/2025 removed file_open_std_files
   06/16/2025 updates for dylib and trampoline merging
   06/24/2025 renamed file_is_valid_name to file_has_valid_dsr_name 
   06/30/2025 added methods for memory access
   07/05/2025 removed strncmp, memset, file_has_valid_dsr_name
   07/08/2025 added k3_sams_get_region_info
   07/12/2025 added shared memory methods
              removed memory prototype methods
   07/14/2025 updated prototype for k3_sams_get_region_info
   08/28/2025 removed mem.h reference
   10/13/2025 removed uint2str
   10/23/2025 added k2_console_display_flash
   12/12/2025 added fopen_pio
*/

#ifndef DYLIB_PRIVATE_H
#define DYLIB_PRIVATE_H

#include <stdio.h>
#include <string.h>
#include <sys/ipc.h>
#include <sys/shm.h>

typedef struct {

   // KERNEL ROM 1

   // stdio methods

   FILE *(*fopen) (const char *name, const char *mode);
   int (*fread) (void *restrict ptr, int size, int nitems, FILE *restrict f);
   int (*fwrite) (const void *restrict ptr, int size, int nitems, FILE * restrict f);
   char *(*fgets) (char *s, int size, FILE *f);
   int (*fputs) (const char *s, FILE *f);
   int (*fseek) (FILE *stream, long offset, int whence);
   long (*ftell) (FILE *stream);
   int (*fclose) (FILE *f);

   // string methods

   int (*strlen) (const char *s);
   char *(*strcpy) (char *d, const char *s);
   int (*strcmp) (const char *s1, const char *s2);
   char *(*strchr) (const char *s, int c);
   int (*strcspn) (char *string, char *chars);
   char *(*strcat) (char *dest, const char *add);
   void *(*memcpy) (void *dest, const void *src, int cnt);
   char *(*strtok) (char *restrict str, const char *restrict sep);

   // conversion methods
 
   char *(*int2str) (int x);

   // stdlib methods

   int (*atexit) (void (*function) (void));
   void (*k1_exit) (int status);
   void (*_Exit) (int status);

   // unistd methods

   char *(*getcwd) (char *buf, int size);

   // internal methods

   char *(*console_gets) (char *buf, int maxlen, bool secure);
   int (*console_puts) (const char *s);
   FILE *(*fopen_pio) (const char *name, const char *mode);

   void (*console_sprite_init) ();
   void (*atexit_init) ();
   void (*sequencer_init) ();
   char *(*proc_get_pid_path) (int pid, char *path);
   void (*generate_unix_path) (char *cwd_path, char *req_path, bool *final_path_ok, char *final_path);
   void (*generate_final_path) (char *final_dsr_path, char *path, bool is_dir);
   FILE *(*file_get_available) (bool is_dsr_file);
   void (*file_atexit) ();

   void (*spinner_read) ();
   void (*spinner_write) ();
   void (*spinner_restore) ();
   unsigned char (*dsrlnk) (struct PAB *pab, unsigned int vdp);
   void (*reboot) ();

   // KERNEL ROM 2

   int (*k2_console_bitmap_load_image) (const char *cache_name, const char *path);
   void (*k2_console_bitmap_draw_image) (const char *cache_name, int y, int x);
   void (*k2_console_display_flash) ();

   void (*k2_proc_get_ea5_params) (char *stdin_fn, char *stdout_fn, char *stderr_fn, int *argc, char *argv[]);

   // KERNEL ROM 3

   void (*k3_sams_get_region_info) (int region, char *name, int *age);

   int (*k3_shmget) (key_t key, long size, int shmflg);
   void *(*k3_shmat) (int shmid, const void *shmaddr, int shmflg, int page);
   int (*k3_shmdt) (const void *shmaddr);
   int (*k3_shmctl) (int shmid, int cmd, struct shmid_ds *buf);

   // INTERNAL DATA

   // Conversion methods

   char *strtok_lasts;
   char num2str_buf[12];

   // 32-bit int div/mod

   long numerator_s32_cache;
   long denominator_s32_cache;
   long quotient_s32_cache;
   long remainder_s32_cache;

} dylib_t;

// initializes the dynamic library 
void dylib_init ();

// sets the dylib links for each rom page
void dylib_rom1_link ();
void dylib_rom2_link ();
void dylib_rom3_link ();

// initializes the dynamic library when an EA5 program is loaded
void dylib_ea5_link ();

#endif
