/* sams_private.h
   This header defines the private methods to support sams memory management.
   change log
   01/18/2025 added header
              added all methods within this header so the methods could be moved to separate files
              removed unused files in sams_cache_t
   01/19/2025 added age to sams_cache_t
   06/24/2025 removed sams_cache_t count element
              added named_page_count to sams_status_t
              removed sams_init_data
   06/25/2025 added and used SAMS_MAX_NAMED_REGIONS
              added SAMS_MAX_PAGES
   06/30/2025 added sams_map_region_and_set_name
   07/05/2025 updated to avoid using lower 16 pages except for 0x2000 and 0x3000
   07/08/2025 added sams_get_region_count, sams_get_region_info, kernel_sams_get_region_info
   07/09/2025 removed pages_in_use, renamed named_page_count to named_region_count, and page_in_use to region_in_use
   07/14/2025 added SAMS_MAX_CACHE_AGE, SAMS_LOCKED_AGE, sams_lock_named, sams_lock_region, sams_unlock_region, sams_set_region_age
*/

#ifndef SAMS_PRIVATE_H
#define SAMS_PRIVATE_H

#include <stdbool.h>
#include <stdio.h>

// #define SAMS_PERFORM_TEST

#define SAMS_MAX_NAMED_REGIONS ((24 * 1024) / (32 + 2))              // 722 is the maximum number of named regions that will fit in a 24KB region located at A000-FFFF
#define SAMS_MAX_PAGES         (SAMS_MAX_NAMED_REGIONS * 6 + 16)     // 4348 is the maximum number of pages that can be supported
#define SAMS_MAX_CACHE_AGE     999
#define SAMS_LOCKED_AGE        0

// sams status in cache ram
typedef struct {
   int region_in_use;
   int page_count;
   int named_region_count;
   int proc_page[6];
} sams_status_t;

// sams info for finding named pages
typedef struct {
   char filename[SAMS_MAX_NAMED_REGIONS][FILENAME_MAX];
   int age[SAMS_MAX_NAMED_REGIONS];
} sams_cache_t;

// EXTERNAL METHODS

// initializes the sams device for use
void sams_init ();

// restores the sams device to its default state
void sams_reset ();

// locates a named cache and reconfigures the sams device to it
int sams_find_cache (const char *name);

// INTERNAL METHODS

// enables the already configured sams device map
void sams_enable_map ();

// disables the sams device map
void sams_disable_map ();

// maps a single page to a given location
void sams_map_page (int page, int location);

// maps a page region (six 4 KB pages in upper ram memory
void sams_map_region (int r);

// counts the number of 4 KB pages in the sams device
int sams_get_page_count ();

// copies a named path to a sams cache
int sams_clone_file (const char *cache_name, unsigned char *sams_addr, const char *path);

// copies sams cached data to vdp
void sams_vdpmemcpy (const char *cache_name, unsigned char *sams_addr, unsigned int vdp_addr, int len);

// expires a sams cache
int sams_expire_cache (const char *name);
int sams_expire_region (int region);

// sets a region age
int sams_set_region_age (int region, int age);

// locks a sams cache / region
int sams_lock_cache (const char *name);
int sams_lock_region (int region);

// unlocks a region
int sams_unlock_region (int region);

// returns the first page associated with a region
int sams_region_to_page (int region);

// returns the number of regions that exist
int sams_get_region_count ();

void kernel_sams_get_region_info (int region, char *name, int *age);
void sams_get_region_info (int region, char *name, int *age);

// sets the (default) process pages 
void sams_save_proc_pages ();

// restores the sams page mapping for the process, which could have non-default mapping
void sams_restore_proc_pages ();

#endif
