/* stdio_private.h
   This header defines the standard i/o private data and methods.
   change log:
   11/23/2023 added header
   12/10/2023 updated to use FILENAME_MAX for the dsr name length
   12/20/2023 added binary file support
   12/22/2023 moved PIO_FILE to public header
   01/22/2024 moved DSR method call typedefs to this file
   02/17/2024 improvements for cached info
   02/24/2024 broad support for shared libraries
   05/03/2025 removed printer file
   06/02/2025 added file_set_handle_executable
   06/24/2025 renamed file_is_valid_name to file_has_valid_dsr_name
              added file_is_valid_path
   10/13/2025 renamed file_has_valid_dsr_name file_get_dsr_cru
*/

#ifndef STDIO_PRIVATE_H
#define STDIO_PRIVATE_H

#include <stdarg.h>
#include <stdio.h>
#include <constants.h>

#define FILE_DSR_MAX             3 // maximum number of native TI-99 files
#define FILE_SPECIAL_MAX         7 // maximum number of special files, such as console I/O. In the future speech, maybe more
#define FILE_ALL_MAX             (FILE_DSR_MAX + FILE_SPECIAL_MAX)
#define FILE_BUFFER_SIZE         0x100
#define FILE_BINARY_BLOCK_SIZE   255 
#define FILE_DATA_VDP_ADDR_FIRST 0x3400
#define FILE_PAB_VDP_ADDR_FIRST  (FILE_DATA_VDP_ADDR_FIRST + FILE_DSR_MAX * FILE_BUFFER_SIZE)
#define FILE_PAB_NAME_MAX_LEN    34

#define FTYPE_NONE         UNDEFINED
#define FTYPE_CONSOLE_OUT  0
#define FTYPE_CONSOLE_IN   1
#define FTYPE_REG_FILE     2
#define FTYPE_PARALLEL_OUT 3
#define FTYPE_PARALLEL_IN  4

// type for the DSR info
typedef struct {
   dsr_state_t dsr[FILE_DSR_MAX];
} dsr_info_t;

// type for all the files
typedef struct {
   FILE file[FILE_ALL_MAX];
} file_info_t;

struct __attribute__((__packed__)) EntryLink {
   struct EntryLink* next;
   unsigned int* routine;
};

struct __attribute__((__packed__)) NameLink {
   struct NameLink* next;
   unsigned int routine;
   char name[8]; // length byte + [upto] 7 characters.
};

struct __attribute__((__packed__)) DeviceRomHeader {
   unsigned char flag;
   unsigned char version;
   unsigned int reserved1;
   struct EntryLink* powerlnk;
   struct NameLink* userlnk;
   struct NameLink* dsrlnk;
   struct NameLink* basiclnk;
   struct EntryLink* interruptlnk;
   unsigned int* gromsomething;
};

// dsr type
typedef struct {
   char name[8];
   int cru;
} dsr_t;

// dsr list type
typedef struct {
   dsr_t dsr[FILENAME_MAX];
   int count;
   bool dual_rs232;
   char root_dsr[FILENAME_MAX];
} dsr_list_t;

// initializes file package
void file_init ();

// waits for the hardware to be ready
void dsr_hardware_wait_for_ready ();

// loads the dsr list
void dsr_list_load ();

// prints the dsr list
void file_print_dsr_list ();

// returns the dsr's cru for a given path
int file_get_dsr_cru (const char *name);

// determines if a file path (in unix format) has a full dsr path
bool file_has_full_dsr_path (const char *name);

// determines if the path is fully valid, testing for a valid full unix path expansion and that the dsr is also valid
// to-do: there's a number of methods that test for path and dsr validity. may be able to combine.
bool file_is_valid_path (const char *path);

// enables a rom by cru
void enable_rom (int crubase);

// disables a rom by cru
void disable_rom (int crubase);

// initializes the standard files
void file_open_std_files (const char *stdin_filename, const char *stdout_filename, const char *stderr_filename);

void file_open_std_files_console_only ();

// shutdown the file package
void file_atexit ();

// get an available file
FILE *file_get_available (bool is_dsr_file);

// initializes a FAB by file index
void file_init_pab (FILE *f);

int file_get_dsr_status (const char *name);

// returns the root filesystem path
const char *get_root_fs ();

FILE *filestream (int fd);

// writes the binary file status block - used by fopen and fclose
int file_write_binary_file_status (FILE *f);

// returns true if the file is executable
bool file_handle_is_executable (FILE *f);
int file_set_handle_executable (FILE *f, bool is_executable);
bool file_is_executable (const char *path);

FILE *fopen_console (const char *mode);
FILE *fopen_pio (const char *name, const char *mode);
FILE *fopen_regular_file (const char *name, const char *mode);

int remove_file (const char *path);

unsigned char dsrlnk_no_copy (unsigned int vdp);

int kernel_vsprintf (char * restrict str, const char * restrict format, va_list argp);

#endif
