/* unistd_private.h
   This header defines the types and methods associated with unistd.
   change log:
   02/24/2024 added header
   04/08/2024 removed get_program_type
   01/12/2025 removed rom_t and rom_cache_init
   02/22/2025 added kernel_proc_get_ea5_params
   05/27/2025 moved dsr_path_is_root, paths_differ, dirname_is_same, dir_entry_count, is_dir, and dir_parent_exists to this package
*/

#ifndef UNISTD_PRIVATE_H
#define UNISTD_PRIVATE_H

#include <stdbool.h>
#include <stdio.h>

#define GPLWSR11  *((volatile unsigned int*)0x83F6)
extern int* exec_prg;

// returns the rom start address for the named rom program
unsigned int get_rom_start_addr (const char *romprogname);

// executes a unix ea5 program
int exec_uea5 (const char *path);

// executes the rom at the specified address
void exec_rom (unsigned int addr);

// the process list type
typedef struct {
   int count;
   int id[16];
} proc_list_t;

// initializes the cache proc list
void proc_cache_init ();

// the login type
typedef struct {
   int user_id;
   char name[10];
   char home_dir[FILENAME_MAX];
   char cwd[FILENAME_MAX];
   char shell[FILENAME_MAX];
   int session_id;
} login_t;

// initializes the login cache
void login_cache_init ();

char *proc_get_pid_path (int pid, char *path);
void proc_exec_next_ea5 ();
void proc_add (const char *stdin_path, const char *stdout_path, const char *stderr_path, int argc, const char *argv[]);
void proc_get_ea5_params (char *stdin_fn, char *stdout_fn, char *stderr_fn, int *argc, char *argv[]);
void kernel_proc_get_ea5_params (char *stdin_fn, char *stdout_fn, char *stderr_fn, int *argc, char *argv[]);

const char *get_unaliased_cmd (const char *short_cmd);

void generate_unix_path (char *cwd_path, char *req_path, bool *final_path_ok, char *final_path);
void generate_dsr_path (char *dsr_path, char *req_path, bool is_dir);
void generate_final_path (char *final_dsr_path, char *path, bool is_dir);
bool dsr_path_is_root (const char *path);
bool paths_differ (const char *path1, const char *path2, bool are_dirs);
bool dirname_is_same (const char *path1, const char *path2);
bool dir_parent_exists (const char *path);
int dir_entry_count (const char *path);
bool is_dir (const char *path);

// determines if a given username/password combination is valid
bool passwd_login_is_valid (char *username, char *password);

// returns the user's password cipher
bool passwd_get_cipher (const char *name, char *cipher);

int passwd_update_cipher (const char *name, const char *passwd_cipher);

#define PASSWD_LOGIN   0
#define PASSWD_CIPHER  1
#define PASSWD_UID     2
#define PASSWD_GID     3
#define PASSWD_NAME    4
#define PASSWD_HOMEDIR 5
#define PASSWD_SHELL   6

bool passwd_get_value (const char *login, int passwd_key, char *value);

const char *get_user_shell_path ();

#endif
