/* stdio.h
   This header defines the methods required for basic standard I/O.
   limitations:
   This implementation only supports text-based file handling.
   change log:
   07/01/2023 initial version
   10/21/2023 removed window references
   11/21/2023 added FILENAME_MAX
   11/22/2023 added feof and fflush
   11/23/2023 split out dsr state from FILE
   12/19/2023 added binary support
   12/22/2023 moved PIO_FILE to this header
   02/16/2024 added write cache handling
   02/24/2024 broad changes for introduction of dylib
   01/20/2025 renamed the device PIO /dev/pio
   05/03/2025 removed PIO_FILE
*/

#ifndef STDIO_H
#define STDIO_H

#include <stdarg.h>
#include <files.h>
#include <stdbool.h>

#define NULL 0x0000

#define FILENAME_MAX 32

// dsr state data
typedef struct {
   struct PAB pab;                     // peripheral address block
   char pab_filename[FILENAME_MAX];    // filename retained for the pab
   unsigned int vdp_data_buffer_addr;  // area in VDP memory where the I/O will be performed
   unsigned int vdp_data_buffer_wp;    // current write position in the VDP memory, used only during writing
   unsigned int vdp_pab_buffer_addr;   // area in VDP memory where the PAB will be copied to
   unsigned int read_write_op;         // cache for retaining whether this FILE is for reading or writing
   bool in_use;                        // indicates whether this block is in use
} dsr_state_t;

// binary file state data
typedef struct {
   long pos;                           // file pointer
   long len;                           // length of the data in the file
   int last_block;                     // last block used in the file
   bool is_executable;                 // indicates whether the binary file is executable
} binary_state_t;

// the FILE type
typedef struct {
   int fd;                             // file descriptor
   int ftype;                          // type of file
   dsr_state_t *dsr;                   // dsr info
   bool is_binary;                     // text vs binary file
   binary_state_t bstate;              // binary file state info
   bool wc_active;                     // write cache active status
   bool eof;                           // indicates eof
   bool in_use;                        // indicates whether this file is in use
} FILE;

extern FILE *stdin;                    // standard input file stream
extern FILE *stdout;                   // standard output file stream
extern FILE *stderr;                   // standard error file stream

// opens a file by name and specified mode;
// text files: r = read, w = write, a = append
// binary files: rb = read, rb+ = read/write, wb = truncate and write, wb+ truncate and read/write
FILE *fopen (const char *name, const char *mode);

// writes the character to the specified file
int fputc (int c, FILE *stream);

// writes a string to the specified file
int fputs (const char *s, FILE *f);

// writes a formatted string using the specified values to the specified file
int fprintf (FILE * restrict f, const char * restrict fmt, ...);

// reads data from the specified file
int fread (void *restrict ptr, int size, int nitems, FILE *restrict f);

// writes data to the specified file
int fwrite (const void *restrict ptr, int size, int nitems, FILE * restrict f);

// seeks the file stream to an offset relative to the whence value, whose options are
#define SEEK_SET 0
#define SEEK_CUR 1
#define SEEK_END 2
int fseek (FILE *stream, long offset, int whence);

// returns the current stream position
long ftell (FILE *stream);

// rewinds the stream to the beginning of the file
void rewind (FILE *stream);

// gets a string from the specified file
char *fgets (char *s, int size, FILE *f);

// flushes the file
int fflush (FILE *f);

// returns positive if file is at eof
int feof (FILE *f);

// closes the specified file
int fclose (FILE *f);

// returns the file descriptor for the stream
int fileno (FILE *stream);

// returns a string formatted per the specified format for the specified values
int sprintf (char *result, const char *fmt, ...);

// returns a string formatted per the specified format for the specified va_list args
int vsprintf (char * restrict str, const char * restrict format, va_list argp);

// writes a character to stdout
int putchar (int c);

// writes a string to stdout
int puts (const char *s);

// reads a string from stdin
char *gets (char *str);

// writes a formatted string using the specified values to stdout
int printf (const char *fmt, ...);

// removes a directory entry
int remove (const char *path);

#endif
