/* util.h
   This header defines useful methods.
   change log:
   07/01/2023 initial version
   11/23/2023 added comments
   12/26/2024 added print
   06/08/2025 added cat_handle
   06/11/2025 changed more def
   06/12/2025 added compare
   10/09/2025 add shared memory management methods
   10/21/2025 added calc_power, calc_shift_for_div_alt and calc_mask_for_mod_alt
   11/22/2025 renmaed compare to compare_file
*/

#ifndef UTIL_H
#define UTIL_H

#include <stdio.h>
#include <stdbool.h>
#include <sys/ipc.h>

// prints the file text to a screen at a time with ability to exit (press c).
// if filename is null, then stdin will be used
void more (const char *filename, bool print_over);

// prints the file text to a screen at a time with ability to exit (press c).
// this version ensures each word appears on a line with no wrap to the next -- much better for reading
void more_words (const char *filename);

// prints the file text to a screen
int cat (const char *filename);
int cat_handle (FILE *f);

// this version ensures each word appears on a line with no wrap to the next -- much better for reading
void cat_words (const char *filename);

// prints the file text to a screen
int catb (const char *filename);

// compares two files
int compare_file (const char *path1, const char *path2);

// copies a text file to another
void fcopy (FILE *fin, FILE *fout);

// returns the size of an object in powers of 2. not exactly high performance but it is called only once, so no big deal
int calc_power (int k);

// returns the shift alternative to division for divisors that are powers of 2
int calc_shift_for_div_alt (int x);

// returns the mask alternative to modulus for divisors that are powers of 2
int calc_mask_for_mod_alt (int s);

// SHARED MEMORY MANAGER

#define SHMMGR_REGION_MAX        3                       // allow up to three regions
#define SHMMGR_ALLOC_MATRIX_SIZE 1024                    // support up to 8192 allocations

typedef struct {
   char name[6];
   key_t key[SHMMGR_REGION_MAX];                         // shared memory keys
   int shmid[SHMMGR_REGION_MAX];                         // shared memory ids

   unsigned char alloc_matrix[SHMMGR_ALLOC_MATRIX_SIZE]; // alloc matrix
   int alloc_matrix_first_free;                          // first matrix element to begin searching

   int cur_region;                                       // currently loaded region
   int cur_page;                                         // currently loaded page

   int region_div;                                       // region divisor, pre-calculated
   int page_div;                                         // page divisor, pre-calculated

   int alloc_size;                                       // power of 2 storage size that fits the requested allocation type size
   int alloc_max;                                        // maximum number of allocations
} shmmgr_t;

// initializes the shared memory tracking structures
int shmmgr_init (shmmgr_t *shmmgr, const char *name, int value_type_size);

// returns an index to the allocated storage
int shmmgr_alloc (shmmgr_t *shmmgr);

// returns a pointer to the value address 
void *shmmgr_map_and_return_ptr (shmmgr_t *shmmgr, int index);
 
// frees an allocation
void shmmgr_free (shmmgr_t *shmmgr, int index);

// terminates the shared memeory tracking
void shmmgr_term (shmmgr_t *shmmgr);

#endif
