/* atof.c
   This method converts an ascii representation of a float to a float.
   change log:
   06/23/2023 initial version
   12/10/2023 changed reference to string.h
*/

#include <stdlib.h>
#include <ctype.h>
#include <string.h>

// converts an ascii representation of a float to a float
double atof (const char *a) {

   // The TI-99/4A uses RADIX-100 for doubles. RADIX is also known as base. So the doubles are 
   // stored in base 100. Doubles are stored as 8 bytes, where the number is in scientific notation,
   // but again as radix 100:
   // - The first byte is the exponent (a radix 100 exponent, not base 10 exponent).
   // - Subsequent bytes store the mantissa. The decimal point is after the first mantissa digit.
   // - Negative numbers are stored with the exponent and first digit negated.
   // - Zero has an exponent and first digit set to 0 and 0, respectively.
   // - Exponents are offset by 0x40.
   // - Note that there are excellent explanations online, but even those have some incorrect info.
   // - https://www.unige.ch/medecine/nouspikel/ti99/reals.htm
   // - For instance, one example gets the matissa correct but the author inadvertantly used base
   // - 10 for the exponent, rather than base 100.

   char *s = (char *) a;
   double d;                // return value
   char * pd = (char *) &d; // pointer to the return value

   // staging values for final double value
   int exp     = 0;                                            // base 100 exponent
   int neg_pos = 1;                                            // pos = 1, neg = -1
   int v[14]   = { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 }; // represented as base 10 later to be compbined as base 100
   int pos     = 0;                                            // write pos in v[]
   int dp      = 0;                                            // decimal point position relative to v[]

   int b10_exp_neg_pos = 1;
   int b10_exp = 0;
   int b10_exp_have_digit = 0;

   int n;
   int have_pos_digit = 0;
   int i;

   // skip whitespace
   s = skip_whitespace (s);

   // capture the negative sign if present
   if (*s == '-') {
      neg_pos = -1;
      s++;
   }

   // run through the digits before any decimal point
   while ( (*s) && (isdigit (*s)) && (pos < 14) ) {
      n = *s;
      n -= 48;
      if (n >= 1) have_pos_digit = 1;
      if (have_pos_digit) {
         v[pos] = n;
         pos++;
      } 
      s++;
   }

   // capture the decimal point position, even if implicit due to lack of one
   if (*s == '.') {
      s++;
   }
   dp = pos;

   // run through the fractional digits before any exponent designation
   while ( (*s) && (isdigit (*s)) && (pos < 14) ) {
      n = *s;
      n = n - '0'; 
      if (n >= 1) have_pos_digit = 1;
      if (have_pos_digit) {
         v[pos] = n;
         pos++;
      } else {
         dp--;
      }
      s++;
   }

   // handle specified exponent
   if ( (*s == 'e') || (*s == 'E') ) {
      s++;
      if (*s == '-') {
         b10_exp_neg_pos = -1;
         s++;
      }
      while ( (*s) && (isdigit (*s)) ) {
         if (b10_exp_have_digit) {
            b10_exp *= 10;
         }
         n = *s;
         b10_exp += n - '0';
         b10_exp_have_digit = 1;
         s++;
      }
      dp += b10_exp * b10_exp_neg_pos;
   }
   
   // process the collected value if at least one positive digit was found, otherwise the value is zero
   if (have_pos_digit) {
      // calculate the exponent, which is base 100
      exp = 0x3F + dp / 2;
      if ( (dp > 0) && (dp % 2) ) exp++;
   
      // shift the base 10 digits right by one if not exp aligned for base 100
      if (dp % 2) {
         for (i = 13; i > 0; i--) {
            v[i] = v[i - 1];
         }
         v[0] = 0;
      }
   }

   // finally, convert the collected and processed value into a double

   // generate the exponent, write and advance. Note that this is a base 100 exponent
   if (neg_pos == -1) exp++;
   n = exp * neg_pos;
   *pd = n;
   pd++;

   // generate the first mantissa digit, that might need to be negated, write and advance
   n = (v[0] * 10 + v[1]) * neg_pos; // base 100 generated by combining the v[0] and v[1] values
   *pd = n;
   pd++;
   
   // process and write the remaining base 100 digits, combining the adjacent base 10 digits
   for (i = 2; i < 14; i += 2) {
      n = v[i] * 10 + v[i + 1];
      *pd = n;
      pd++;
   }

   return d;
}
