#include <cipher.h>
#include <cipher_private.h>
#include <string.h>
#include <dylib.h>

/*
VIC Cipher - POLYALPHABETIC SUBSTITUTION CIPHER
Used by Soviet spies all over the world, in the middle of the twentieth century. Its name is based on a nickname VICTOR of a Soviet agent spying in USA under the name Reino Häyhänen. In 1957 he surrendered to the American intelligence and disclosed details of the cipher.

Usage
The VIC cipher is regarded as the most complex modification of the Nihilist cipher family. It is considered to be one of the strongest ciphers, which can be used manually without computers. By the time it was disclosed as a result of betrayal, American counterintelligence hadn't managed to break the cipher.

Algorithm
The VIC cipher uses a table which allows changing letters of plaintext into numbers. It is called a straddling checkerboard.

It differs from tables used in other substitution ciphers because it produces shorter sequences of numbers (it is much more comfortable for sending to the second party).

This implementation is intended to be map all ASCII codes, rather than numbers and letters, thus allowing upper and lower case as
well as special characters which are commonly required in passwords. Further the range of acceptable cipher values must be 
constrained to that which is printable and has a known range that can be embedded within other strings and can be extracted with
methods such as strtok.

The exact algorithm is:
- mapping of all nibble values in grid of 1 x 16
- convert ASCII to hex values (nibbles)
- capture the location of the nibble values within the map
- convert the location of the nibble values into textual hex

The output for any ASCII value is 2 textual values in the range 0-9 and A-F.
*/

// the cipher vic mapping
const unsigned char cipher_vic_map[16] = 
   { 0x0f, 0x03, 0x0a, 0x07, 0x02, 0x0b, 0x0d, 0x09, 0x05, 0x04, 0x0c, 0x0e, 0x00, 0x06, 0x01, 0x08 };

// encrypts the plaintext into the vic cipher
int cipher_vic_encrypt (const char *plaintext, char *cipher, int len) {

   int r;                                          // the return result
   int i, j;                                       // loop vars
   char vh, vl;                                    // the cipher high and low nibble currently being processed

   const int plaintext_len = dylib.strlen (plaintext);   // get the plaintext len

   if (plaintext_len * 2 < len - 1) {              // ensure the cipher will fit in the provided buffer
   
      for (i = 0; i < plaintext_len; i++) {        // loop through all the plaintext
         vh = plaintext[i] >> 4;                   // capture the current character high and low nibble values
         vl = plaintext[i] & 0x0f;
   
         for (j = 0; j < 16; j++) {                // locate the high bit nibble mapping and convert to ASCII hex representation
            if (vh == cipher_vic_map[j]) {
               if (j <= 9) {
                  *cipher = '0' + j;               // save the cipher high nibble
               } else {
                  *cipher = 'A' + j - 10;          // ditto
               }
               cipher++;                           // advance to the next position in the cipher
               break;                              // break out of the loop
            }
         }
   
         for (j = 0; j < 16; j++) {                // locate the low bit nibble mapping and convert to ASCII hex representation
            if (vl == cipher_vic_map[j]) {
               if (j <= 9) {
                  *cipher = '0' + j;               // save the cipher low nibble
               } else {
                  *cipher = 'A' + j - 10;          // ditto
               }
               cipher++;                           // advance to the next position in the cipher
               break;                              // break out of the loop
            }
         }

      }
      *cipher = 0x00;                              // null terminate the cipher

      r = 0;                                       // set the return value to success
   } else {
      r = -1;                                      // set the return value to failed
   }

   return r;                                       // return the result
}
