/* compare_file.c
   This method compares two files.
   change history
   06/12/2025 initial version
   11/22/2025 renamed to compare_file
*/

#include <util.h>
#include <sys/stat.h>
#include <dylib.h>
#include <stdio_private.h>

void write_file_not_found (const char *name) {
   dylib.fputs ("diff: ", stderr);
   dylib.fputs (name, stderr);
   dylib.fputs (": No such file or directory\n", stderr);
}

void write_is_a_dir (const char *name) {
   dylib.fputs ("diff: ", stderr);
   dylib.fputs (name, stderr);
   dylib.fputs (": Is a directory\n", stderr);
}

void write_file_types_differ (const char *name1, const char *name2) {
   dylib.fputs ("diff: ", stderr);
   dylib.fputs (name1, stderr);
   dylib.fputs (" and ", stderr);
   dylib.fputs (name2, stderr);
   dylib.fputs (": file types differ\n", stderr);
}

void write_file_open_failure (const char *name) {
   dylib.fputs ("diff: ", stderr);
   dylib.fputs (name, stderr);
   dylib.fputs (": file open failure\n", stderr);
}

void write_unhandled_file_type (const char *name1, const char *name2) {
   dylib.fputs ("diff: ", stderr);
   dylib.fputs (name1, stderr);
   dylib.fputs (" and ", stderr);
   dylib.fputs (name2, stderr);
   dylib.fputs (": unhandled file type\n", stderr);
}

void write_binary_files_differ (const char *name1, const char *name2) {
   dylib.fputs ("Binary files ", stderr);
   dylib.fputs (name1, stderr);
   dylib.fputs (" and ", stderr);
   dylib.fputs (name2, stderr);
   dylib.fputs (" differ\n", stderr);
}

void write_text_files_differ (const char *name1, const char *name2) {
   dylib.fputs ("Text files ", stderr);
   dylib.fputs (name1, stderr);
   dylib.fputs (" and ", stderr);
   dylib.fputs (name2, stderr);
   dylib.fputs (" differ\n", stderr);
}

int compare_file (const char *path1, const char *path2) {

// to-do: refactor this code to test the non-file conditions in a flat way -- way too much hierchical conditional tests

   const char *read_mode[]  = {"", "", "r", "rb", "", ""};

   struct stat stat1, stat2;

   FILE *f1, *f2;
   char s1[256], s2[256];
   char *q1, *q2;
   int c1, c2, i;

   int r = 0;

   int r1 = stat (path1, &stat1);
   int r2 = stat (path2, &stat2);

   if (!r1) {
      if (!r2) {
         if (S_ISREG (stat1.st_mode)) {
            if (S_ISREG (stat2.st_mode)) {
               if (stat1.st_mode == stat2.st_mode) {

                  // open files by type
                  f1 = dylib.fopen (path1, read_mode[stat1.st_mode]);
                  if (f1) {
                     f2 = dylib.fopen (path2, read_mode[stat2.st_mode]);
                     if (f2) {
                        switch (stat1.st_mode) {
                           case 2:
                              while (1) {
                                 q1 = dylib.fgets (s1, sizeof (s1), f1);
                                 q2 = dylib.fgets (s2, sizeof (s2), f2);
                                 if ((q1 && !q2) || (!q1 && q2)) {
                                    r = 1;                          // files have different line counts
                                 } else {
                                    if (dylib.strcmp (q1, q2)) {
                                       r = 1;                       // text on line differs
                                    }
                                 }
    
                                 if (r) {
                                    write_text_files_differ (path1, path2);
                                 }
                                 if (r || (q1 == q2)) {             // break out if a difference was detected, or both q1 and q2 are equal (both will be NULL if so)
                                    break;
                                 }
                              }
                              break;
                           case 3:
                              while (1) {
                                 c1 = dylib.fread (s1, 1, FILE_BINARY_BLOCK_SIZE, f1);
                                 c2 = dylib.fread (s2, 1, FILE_BINARY_BLOCK_SIZE, f2);
                                 if (c1 != c2) {
                                    r = 1;                          // counts differ
                                 } else {
                                    for (i = 0; i < c1; i++) {
                                       if (s1[i] != s2[i]) {
                                          r = 1;                    // bytes differ
                                          break;
                                       }
                                    }
                                 }
                                 if (r) {                           // break if a difference was detected
                                    write_binary_files_differ (path1, path2);
                                 }
                                 if (r || (c1 <= 0) || (c2 <= 0)) {
                                    break;
                                 }
                              }
                              break;
                           default:
                              write_unhandled_file_type (path1, path2);
                              r = 1;                                // unhandled file type
                              break;
                        }
                        dylib.fclose (f2);
                     } else {
                        write_file_open_failure (path2);
                        r = 1;                                      // cannot open path2
                     }
                     dylib.fclose (f1);
                  } else {
                     write_file_open_failure (path1);
                     r = 1;                                         // cannot open path1
                  }
               } else {
                  write_file_types_differ (path1, path2);
                  r = 1;                                            // file types differ
               }
            } else {
               write_is_a_dir (path2);
               r = 1;                                               // r2 is a directory
            }
         } else {
            write_is_a_dir (path1);
            r = 1;                                                  // r1 is a directory
         }   
      } else {
         write_file_not_found (path2);
         r = 1;                                                     // r2 doesn't exist
      }
   } else {
      write_file_not_found (path1);
      r = 1;                                                        // r1 doesn't exist
   }

   return r;                                                        // return the result
}
