/* console_display_set_mode.c
   This method provides a single way to set the display mode.
   change log:
   07/23/2023 initial version
   08/11/2023 modified to use mode from system cache
   08/11/2023 moved extern reference to wd to private include file
   08/11/2023 removed clearing of the screen
   08/11/2023 removed blanking of the screen
   08/11/2023 removed loading standard fonts in this method
   08/18/2023 minor updates to comments
   09/27/2023 added sprite initialization
   10/20/2023 modified to use cache values
   10/21/2023 removed references to windows
   10/21/2023 removed use of memory library
   01/12/2024 added F18A unlocking
   01/13/2024 added setting of the SAL when the F18A 80-column mode is selected. Surprisingly this mode supports sprites,
              although unix99r2 does not yet. The SAL and CT were moved as they were overlapping with the SIT in 80 column mode.
   02/14/2024 updated to use SIT frames
              moved CT to behind SIT frame 1 so consolidate VDP usage
   02/16/2024 updated to set the CT back to 0x40
   02/24/2024 modified to use cache and dylib
   12/31/2024 added support for graphics ii/bitmap mode
   01/11/2025 added sprite support for F18 80 column text mode
              updated to use vdp reg 1 definitions from console_private.h
   01/22/2025 performance improvements
   02/08/2025 added GPU initialization and all that entails
   02/10/2025 updated to initialize GPU ahead of setting modes (the tests require use of regs 0x36 and 0x37 which also map
              to the standard 9918A VDP's regs 6 and 7 which mess with the SDT and color table).
              updated to set SDT for text40 mode--shouldn't hurt that the register receives value it will not use with standard
              9918A, but will be used when F18A is present
   02/11/2025 removed use of sit_frame_index
   02/16/2025 migrated gpu common methods to be added into this method
   02/18/2025 updated to send gpu method address first and region size to gpu_init
   02/22/2025 removed commented out code from gpu method mods
   02/28/2025 changes for added rows as a parameter
*/

#include <vdp.h>
#include <console.h>
#include <console_private.h>
#include <cache_private.h>
#include <dylib.h>
#include <gpu_private.h>

__attribute__((noinline)) void gpu_f18a_cls (void) {
   __asm__(
      "cls:\n"
      "       li r0,>2002\n"        // load the address of the rows value
      "       li r1,>2004\n"        // load the address of the cols value
      "       mov *r0,r2\n"         // move the value pointed at by r0 (rows) to r2
      "       mpy *r1,r2\n"         // multiply the value pointed to at by r1 (cols) 
                                    // with r2 and place the result in r2 (msb) and r3 (lsb). in this case our result is in r3 and r2 is 0
      "       sra r3,1\n"           // divide by 2 via shift right 1
      "       mov r3,r2\n"          // move the result back to r2
      "       li r0,>2020\n"        // load a pattern, double spaces, into r0
      "       li r1,>0000\n"        // set the initial address
      "write_all_do:\n"
      "       mov r0,*r1+\n"        // write the contents of r0 to the address at r1 and increment r1
      "       dec r2\n"             // decrement the counter
      "       jne write_all_do\n"   // loop until done
      "       idle\n"               // idle the GPU
      "       end\n"                // should never be reached
      :::
      "r0","r1","r2","r3"
   );
}

__attribute__((noinline)) void gpu_f18a_scroll (void) {
   __asm__(
      "x_scroll:\n"                 // set scrolling data
      "       li r0,>2002\n"        // load the address of the rows value
      "       li r1,>2004\n"        // load the address of the cols value
      "       mov *r0,r2\n"         // move the value pointed at by r0 (rows) to r2
      "       mpy *r1,r2\n"         // multiply the value pointed to at by r1 (cols)
                                    // with r2 and place the result in r2 (msb) and r3 (lsb). in this case our result is in r3 and r2 is 0
      "       sra r3,1\n"           // divide by 2 via shift right 1
      "       mov r3,r2\n"          // move the result back to r2. r2 now has the screen size
      "       mov *r1,r0\n"         // move the columns value into r0
                                    // r0: source address
      "       li r1,>0000\n"        // r1: target address
      "       mov r0,r3\n"          // copy the source (same as width) to r3
      "       sra r3,1\n"           // divide r3 by 2
      "       s r3,r2\n"            // subtract r3 from r2
                                    // r2: words to move
      "x_scroll_do:\n"              // do ... while (count)
      "       mov *r0+,*r1+\n"      // copy word source to target and increment both r0 and r1
      "       dec r2\n"             // decrement r2 count
      "       jne x_scroll_do\n"    // while r2 count, continue looping
      "x_fill:\n"                   // set filling data
      "       li r0,>2020\n"        // r0: value of two spaces (0x20 and 0x20)
                                    // r1: target address (generated from above)
      "       li r3,>2004\n"        // address for screen width
      "       mov *r3,r2\n"         // r2: total bytes to clear
      "       sra r2,1\n"           // r2: divide by 2 - total words to clear
      "x_fill_do:\n"                // do ... while (count)
      "       mov r0,*r1+\n"        // copy the r0 two spaces to target in r1 and increment r1
      "       dec r2\n"             // decrement r2 count
      "       jne x_fill_do\n"      // while r2 count, continue looping
      "x_done:\n"                   // method is done
      "       idle\n"               // idle the GPU
      "       end\n"                // should never be reached
      :::
      "r0","r1","r2","r3"
   );
}

void console_display_set_mode (int display_mode, int rows) {

   // cache the requested display mode
   cachex.console.mode = display_mode;

   // collect the values to be used to configure graphics
   int vdpreg0, vdpreg1, vdpreg1_final_or;
   unsigned char v_sit = 0x00, v_ct = 0x00, v_pdt = 0x00, v_sal = 0x00, v_sdt = 0x00;
   switch (cachex.console.mode) {
      case DISPLAY_MODE_STANDARD:
         cachex.console.screen_height = rows;
         cachex.console.screen_width  = 32;
         vdpreg0          = 0;
         vdpreg1          = DISPLAY_MODE_STANDARD_VDP_REG1;
         vdpreg1_final_or = VDP_SPR_8x8;
         v_sit            = 0x00;
         v_ct             = 0x2c;
         v_pdt            = 0x02;
         v_sal            = 0x1e;
         v_sdt            = 0x02;
         break;
      case DISPLAY_MODE_TEXT:
         cachex.console.screen_height = rows;
         cachex.console.screen_width  = 40;
         vdpreg0          = 0;
         vdpreg1          = DISPLAY_MODE_TEXT_VDP_REG1;
         vdpreg1_final_or = 0;
         v_sit            = 0x00;
         v_pdt            = 0x02;
         v_sal            = 0x1e;
         v_sdt            = 0x02;
         break;
      case DISPLAY_MODE_F18A_TEXT80:
         cachex.console.screen_height = rows;
         cachex.console.screen_width  = 80;
         vdpreg0          = VDP_MODE0_80COL;
         vdpreg1          = DISPLAY_MODE_F18A_TEXT80_REG1;
         vdpreg1_final_or = VDP_SPR_8x8;
         v_sit            = 0x00;
         v_pdt            = 0x02;
         v_sal            = 0x1e;
         v_sdt            = 0x02;
         break;
      case DISPLAY_MODE_BITMAP:
         cachex.console.screen_height = rows;
         cachex.console.screen_width  = 32;
         vdpreg0          = DISPLAY_MODE_BITMAP_REG0;
         vdpreg1          = DISPLAY_MODE_BITMAP_REG1;
         vdpreg1_final_or = VDP_SPR_8x8;
         v_sit            = 0x0E;
         v_ct             = 0xFF;
         v_pdt            = 0x03;
         v_sal            = 0x76;
         v_sdt            = 0x03;
         break;
      default:
         vdpreg0          = 0;
         vdpreg1          = 0;
         vdpreg1_final_or = 0;
         break;
   }

   // enable the F18A if available
   VDP_SET_REGISTER (0x39, 0x1c);  // Write once
   VDP_SET_REGISTER (0x39, 0x1c);  // Write twice, unlock

   // define the location for gpu methods in vdp ram based on the requested display mode
   unsigned int gpu_method_addr_first;
   unsigned int gpu_method_region_size;
   switch (cachex.console.mode) {
      case DISPLAY_MODE_STANDARD:
      case DISPLAY_MODE_TEXT:
      case DISPLAY_MODE_F18A_TEXT80:
         gpu_method_addr_first  = 0x2020;  // first 0x20 bytes at 0x2000 reserved for parameterized methods
         gpu_method_region_size = 0x13E0;
         break;
      case DISPLAY_MODE_BITMAP:
         gpu_method_addr_first  = 0x1820;  // first 0x20 bytes at 0x1800 reserved for parameterized methods
         gpu_method_region_size = 0x07E0;
         break;
      default:
         gpu_method_addr_first  = 0x0000;  // should not be reached
         gpu_method_region_size = 0x0000;
         break;
   }

   // attemp to initialize the gpu (return is 0 for success, 1 for failed), negate result to have a positive value when it exists
   int have_gpu = !gpu_init (gpu_method_addr_first, gpu_method_region_size);

   // add gpu methods and their parameters if the gpu exists
   if (have_gpu) {

      unsigned char *v;

      // load common gpu methods corresponding to the requested mode
      switch (cachex.console.mode) {
         case DISPLAY_MODE_STANDARD:
         case DISPLAY_MODE_TEXT:
         case DISPLAY_MODE_F18A_TEXT80:

            // write the screen height into the VDP as a GPU parameter
            v = (unsigned char *) &cachex.console.screen_height;
            vdpmemset (0x2002, *v, 1);
            v++;
            vdpmemset (0x2003, *v, 1);
         
            // write the screen width into the VDP as a GPU parameter
            v = (unsigned char *) &cachex.console.screen_width;
            vdpmemset (0x2004, *v, 1);
            v++;
            vdpmemset (0x2005, *v, 1);

            gpu_load_method (GPU_CLS, (unsigned char*) gpu_f18a_cls, 0x22);
            gpu_load_method (GPU_SCROLL, (unsigned char*) gpu_f18a_scroll, 0x38);
    
            break;

         case DISPLAY_MODE_BITMAP:
          
            break;
     
         default:
            break;
      }

      gpu_set_methods_watermark ();  
   }

   // move the vdp to this mode
   VDP_SET_REGISTER (VDP_REG_MODE0, vdpreg0);
   VDP_SET_REGISTER (VDP_REG_MODE1, vdpreg1); 

   // handle 30 row mode available only with F18A
   if (have_gpu) {
      if (cachex.console.screen_height == 30) {
         VDP_SET_REGISTER (0x31, 0x40); // set 30 row mode
      } else {
         VDP_SET_REGISTER (0x31, 0x00); // set to standard 24 row mode
      }
   }

   // set up the Screen Image Table (SIT)
   VDP_SET_REGISTER (VDP_REG_SIT, v_sit);
   cachex.console.vdp_screen_image_table    = v_sit * 0x0400;
   cachex.console.vdp_text_addr_current     = cachex.console.vdp_screen_image_table;
   cachex.console.vdp_text_addr_scroll_reqd = cachex.console.vdp_text_addr_current +
                                              cachex.console.screen_height * cachex.console.screen_width;

   // set up the character Pattern Definition Table (PDT)
   VDP_SET_REGISTER (VDP_REG_PDT, v_pdt);
   switch (cachex.console.mode) {
      case DISPLAY_MODE_STANDARD:
      case DISPLAY_MODE_TEXT:
      case DISPLAY_MODE_F18A_TEXT80:
         cachex.console.vdp_character_pattern_table = (unsigned int) v_pdt * 0x0800;
         break;
      case DISPLAY_MODE_BITMAP:
         // need fix: calculate address for this special mode
         cachex.console.vdp_character_pattern_table = 0x0000;
         break;
      default:
         break;
   }

   // set up the Color Table (CT)
   switch (cachex.console.mode) {
      case DISPLAY_MODE_STANDARD:
         // set up the Color Table
         VDP_SET_REGISTER (VDP_REG_CT, v_ct);
         cachex.console.vdp_color_table = (unsigned int) v_ct * 0x0040;
         break;
      case DISPLAY_MODE_BITMAP:
         VDP_SET_REGISTER (VDP_REG_CT, v_ct);
         // need fix: calculate address for this special mode
         cachex.console.vdp_color_table = 0x2000;
         break;
      default:
         break;
   }

   // set up the Sprite Attribute List and Sprite Pattern Definition tables
   VDP_SET_REGISTER (VDP_REG_SAL, v_sal);
   cachex.console.vdp_sprite_attribute_table = (unsigned int) v_sal * 0x0080;

//PRI_5 ? this vdpchar call seems to be unneeded -- console_sprite_init () should take care of it
// vdpchar (cachex.console.vdp_sprite_attribute_table, 0xd0);
 
   VDP_SET_REGISTER (VDP_REG_SDT, v_sdt);
   cachex.console.vdp_sprite_pattern_table = (unsigned int) v_sdt * 0x0800;

   // initialize sprite caches in CPU RAM and in VDP
   if (dylib.console_sprite_init) {
      dylib.console_sprite_init ();
   }

   // unblank, turn on interrupts and set up anything else (sprite mode)
   // vdpreg1 |= VDP_MODE1_UNBLANK | VDP_MODE1_INT | vdpreg1_final_or;
   vdpreg1 |= VDP_MODE1_INT | vdpreg1_final_or;
   VDP_SET_REGISTER (VDP_REG_MODE1, vdpreg1);
   VDP_REG1_KSCAN_MIRROR = vdpreg1;
}
