/* console_getc.c
   This method reads from the keyboard waiting for a new character to be read. When
   no characters are being read, the sequencer is called to update sound and speech.
   change log:
   06/23/2023 initial version
   04/24/2025 updated to use tipi-based external keyboard
   05/15/2025 handled key_up by excluding it
   06/05/2025 updates to send signals when Ctrl-C pressed, for either the console keyboard or an external TIPI keyboard
   06/07/2025 revamped the console keyboard read to be more responsive to the user. The previous method would wait until the keyboard registered no keys,
              and then waited for a new key. The new will accept even the first key found as long as it is different. If the same, it requires one cycle
              of no key press, which should work well. The time required to double hit the same key allows for multiple cycles thankfully.
   06/09/2025 handle ctrl-d
*/

#include <vdp.h>
#include <kscan.h>
#include <console.h>
#include <sequencer_private.h>
#include <keyboard_private.h>
#include <cache_private.h>

int kscan_last_key = 0xff;

int console_getc () {
   int kscan_key;
   int extkb_key;

   // select from the available keyboards
   if (cachex.have_external_keyboard) {

      // read a key, calling the sequencer until it is read.
      // the key value read is ASCII. It will be zero when no key is in process, < 0 when a key is held (negative ASCII), and > 0 when a key is first pressed
      while (1) {
         extkb_key = keyboard_get ();
         if (extkb_key > 0) {
            if (extkb_key == 3) { // ctrl-c
               raise (SIGINT);
            } else {
               break;
            }
         }
         sequencer_update ();
      }
      kscan_key = extkb_key;

   } else {

      while (1) {                                                      // loop until a new key stroke is read
         kscan_key = kscan (5);                                        // read the keyboard
         if ((kscan_key != kscan_last_key) && (kscan_key != 0xff)) {   // accept only if different than the last read and is a key press
            if (kscan_key >= 128) {
               while (kscan (5) != 0xff) {                             // hold until the key is lifted
                  sequencer_update ();
               }

               if (kscan_key == 131) {                                 // capture ctrl-c
                  raise (SIGINT);                                      // raise the signal but do not exit
               } else if (kscan_key == 132) {                          // capture ctrl -d
                  break;                                               // exit
               }

            } else {
               kscan_last_key = kscan_key;                             // record the last key
               break;                                                  // exit the loop with a new key
            }
         } 
         kscan_last_key = kscan_key;                                   // save the current key into the last
         sequencer_update ();                                          // run the sequencer
      }
   }

   return kscan_key;
}
