/* console_gets.c
   This method reads user data from the screen and provides 4-way cursor control
   change log:
   10/20/2023 initial version
   10/22/2023 updated to use new public methods for sequencer
   11/08/2023 added secure option so that passwords can be entered
   12/10/2023 changed reference to math.h
   02/24/2024 modified to use cache
   03/01/2024 modified to use dylib
   01/22/2025 performance improvements
   06/09/2025 added ctrl-d processing
   07/05/2025 dylib adjustments
*/

#include <console.h>
#include <console_private.h>
#include <cache_private.h>
#include <stdio.h>
#include <vdp.h>
#include <sequencer.h>
#include <math.h>
#include <string.h>
#include <dylib.h>

char cursor_char = 30;
char cursor_under_char;
unsigned int cursor_addr;
int cursor_blink_cycle;

void cursor_write_value () {
   VDP_SET_ADDRESS_WRITE (cursor_addr); 
   VDPWD = cursor_under_char;
}

void cursor_place_cursor () {
   VDP_SET_ADDRESS_WRITE (cursor_addr);
   VDPWD = cursor_char;
}

void cursor_read_value () {
   VDP_SET_ADDRESS (cursor_addr);
   cursor_under_char  = VDPRD;
}

void cursor_blink () {
   switch (cursor_blink_cycle) {
      case 0:
         cursor_place_cursor ();
         break;
      case 128:
         cursor_write_value ();
         break;
      case 256:
         cursor_blink_cycle = -1;
         break;
      default:
         break;
   }
   cursor_blink_cycle++;
}

void cursor_blink_initiate (unsigned int addr) {
   cursor_addr = addr;
   cursor_blink_cycle = 0;
   cursor_read_value ();
   sequencer_add (cursor_blink);
}

void cursor_blink_terminate () {
   sequencer_remove (cursor_blink);
   cursor_write_value ();
}

char *console_gets (char *s, int maxlen, bool secure) {
   char *r = s;          // the result
   char secure_buf[32];
   unsigned int start_addr   = cachex.console.vdp_text_addr_current;
   unsigned int current_addr = cachex.console.vdp_text_addr_current;
   unsigned int last_addr    = cachex.console.vdp_text_addr_current;
   int key;
   bool do_continue;
 
   // test the inputs and confirm they're valid
   if (!s || maxlen < 2) {
      do_continue = false;
      r           = NULL;
   } else {
      do_continue = true;
      r           = s;
   }

   if (secure) {
      memset (secure_buf, 0x00, sizeof (secure_buf));
   }

   // process until enter is pressed  
   while (do_continue) {

      cursor_blink_initiate (current_addr);

      key = console_getc ();

      cursor_blink_terminate ();

      // handle keys
      if (key >= 32 && key <= 127) {
         // handle printable keys

         // write this key to the display
         VDP_SET_ADDRESS_WRITE (current_addr);
         if (secure) {
            VDPWD = '*';
            secure_buf[current_addr - start_addr] = key;
         } else {
            VDPWD = key;
         }

         // advance the write position
         current_addr++;

         // check constraints and perform scrolling if needed

         // test if the length so far is ok
         if (current_addr - start_addr < maxlen - 1) {
            // determine if scrolling is needed with current_addr off the bottom of the screen
            if (current_addr >= cachex.console.vdp_text_addr_scroll_reqd) {
               // need to scroll
   
               // confirm the start address isn't already on the first line
               if (start_addr >= cachex.console.screen_width) {
   
                  // scroll
                  int offset = console_scroll_one ();

                  // update values based on the scroll that just occured
//                start_addr   -= cachex.console.screen_width;
                  start_addr   = start_addr + offset - cachex.console.screen_width;
//                last_addr    -= cachex.console.screen_width;
                  last_addr    = last_addr + offset - cachex.console.screen_width;
//                current_addr  = cachex.console.vdp_text_addr_scroll_reqd - cachex.console.screen_width;
                  current_addr = cachex.console.vdp_text_addr_scroll_reqd - cachex.console.screen_width;
               } else {
                  // the start address was already on the first row so cannot scroll any more
                  current_addr--;
               }
            }
         } else {
            // the max length has been reached so don't advance the cursor
            current_addr--;
         }

         // capture the max address on the screen the input has occured on
         last_addr = max (last_addr, current_addr);

      } else {
         switch (key) {
            case 8: // left
               // shift left if the new position will not move before the start position
               if (current_addr - 1 >= start_addr) {
                  current_addr--;
               }
               break;
            case 9: // right
               // shift right if the new position will not move before the last position
               if (current_addr < last_addr) {
                  current_addr++;
               }
               break;
            case 10: // down
               current_addr += cachex.console.screen_width;
               current_addr = min (current_addr, last_addr);
               break;
            case 11: // up 
               if (current_addr < cachex.console.screen_width) {
                  current_addr = start_addr;
               } else {
                  current_addr -= cachex.console.screen_width;
                  current_addr = max (current_addr, start_addr);
               }
               break;
            case 13: // enter
               // done with input
               do_continue = false;
               break;
            case 4: // ctrl-D
            case 132:
               if (!secure && (start_addr == last_addr)) {
                  do_continue = false;
                  r           = NULL;
               }
               break;
            default:
               break;
         }
      }
   }

   if (r) { // r should be set to the address of the input string, unless EOF was triggered via ctrl-d
      // collect the input
      if (secure) {
         dylib.memcpy (s, secure_buf, last_addr - start_addr + 1);
      } else {
         vdpmemread (start_addr, (unsigned char *)s, last_addr - start_addr + 1);
      }
      char *p = s;
      for (int i = 0; i < last_addr - start_addr + 1; i++) {
         if (*p > 32) {
            s = p;
            s++;
         }
         p++;
      }
      *s = 0x00;
   
      // set the current cursor position and write a carriage return
      cachex.console.vdp_text_addr_current = last_addr;
      console_puts ("\n");
   }

   return r;             // return the result
}
