/* console_puts.c
   This method performs the puts function which writes text to the screen and handles a simple set of control characters
   change log:
   10/20/2023 initial version
   12/10/2023 changed reference to math.h
   02/24/2024 modified to use cache
   03/01/2024 modified to use dylib
   12/28/2024 corrected tab position calculation
              updated calculations for carriage return, line feed, backspace and tab to handle any image table location
   01/22/2025 performance improvements
*/

#include <console.h>
#include <console_private.h>
#include <cache_private.h>
#include <constants.h>
#include <math.h>
#include <vdp.h>
#include <dylib.h>

#define TAB_STOP 5

int console_puts (const char *s) {

   char *p_text = (char *)s;    // pointer to chase the current processing point in s
   int n_text       = 0;        // number of bytes to be processed in s this cycle
   int n_write;                 // number of bytes to be processed in a this subcycle due to need for scrolling
   int n_max_before_scroll;     // number of bytes remaining before a scroll is necessary
   unsigned int line_begin;     // address to the beginning of a line
   bool do_continue = true;     // control boolean indicating whether to continue; true until a null character is found
   int r            = 0;        // the return value

   do {                         // cycle until done
      if (*s >= 32) {           // detect printable characters
         n_text++;              // count the number of printable characters in this cycle
      } else {                  // handle non-printable characters
         while (n_text) {       // handle captured printable characters before handling any printable characters counted
            n_max_before_scroll =                                    // calculate num chars that can be printed before scrolling rqd
               cachex.console.vdp_text_addr_scroll_reqd - 
               cachex.console.vdp_text_addr_current;
            n_write             = min (n_text, n_max_before_scroll); // determine the number of bytes to write
            vdpmemcpy                                                // write the bytes to vdp ram
               (cachex.console.vdp_text_addr_current, 
                (unsigned char *)p_text, 
                n_write);
            p_text += n_write;                                       // advance the pointer within s
            n_text -= n_write;                                       // decrement the number of bytes to be written to vdp ram
            cachex.console.vdp_text_addr_current += n_write;         // advance the vdp write address
            if (cachex.console.vdp_text_addr_current >=              // determine if scrolling is required
                cachex.console.vdp_text_addr_scroll_reqd) {
               console_scroll_one ();                                // scroll
            }
         }
         switch (*s) {                                               // process non-printable characters
            case 0:                                                  // null terminator
               do_continue = false;                                  // set for exiting
               break;
            case '\n':                                               // line feed
               cachex.console.vdp_text_addr_current +=               // advance the write address to the next line
                  cachex.console.screen_width - 
                  (cachex.console.vdp_text_addr_current -
                   cachex.console.vdp_screen_image_table) % 
                  cachex.console.screen_width;
               if (cachex.console.vdp_text_addr_current >=           // determine if scrolling is required
                   cachex.console.vdp_text_addr_scroll_reqd) {
                  console_scroll_one ();                             // scroll
               }
               p_text++;                                             // advance past this control character
               break;
            case '\r':                                               // carriage return
               cachex.console.vdp_text_addr_current -=               // move write position back to the beginning of the line
                  (cachex.console.vdp_text_addr_current -
                   cachex.console.vdp_screen_image_table) % 
                  cachex.console.screen_width;
               p_text++;                                             // advance past this control character
               break;
            case '\b':                                               // backspace
               if (cachex.console.vdp_text_addr_current) {           // protect against underflow when write pos is in first pos
                  line_begin =                                       // calculate the beginning of the line
                     cachex.console.vdp_text_addr_current - 
                     (cachex.console.vdp_text_addr_current -
                      cachex.console.vdp_screen_image_table) % 
                     cachex.console.screen_width;
                  cachex.console.vdp_text_addr_current--;            // move back one position
                  cachex.console.vdp_text_addr_current =             // ensure the backspace didn't move before the first pos
                     max 
                        (cachex.console.vdp_text_addr_current, 
                         line_begin);
               }
               p_text++;                                             // advance past this control character
               break;
            case '\f':                                               // form feed
               console_cls ();                                       // clear the screen
               p_text++;                                             // advance past this control character
               break;
            case '\t':                                               // tab
               cachex.console.vdp_text_addr_current +=               // move to the next tab position
                  TAB_STOP - 
                  ((cachex.console.vdp_text_addr_current -
                    cachex.console.vdp_screen_image_table) % 
                   cachex.console.screen_width) %
                  TAB_STOP;
               if (cachex.console.vdp_text_addr_current >=           // determine if scrolling is needed
                   cachex.console.vdp_text_addr_scroll_reqd) {
                  console_scroll_one ();                             // scroll
               }
               p_text++;                                             // move past this control character
               break;
            default:                                                 // unhandled
               r = UNDEFINED;                                        // set an error
               break;
         }
      }
      s++;                                                           // advance to the next position in s
   } while (do_continue);                                            // loop if null wasn't yet found

   return r;                                                         // return the result
}
