/* console_sprite_update.c
   This method performs sprite motion updates and is called by the sequencer.
   change log:
   09/10/2023 initial version
   09/14/2023 updated to use SPRITE ROW/COL MIN values
   09/27/2023 updated to only process sprite motion when the display is in standard (graphics) mode
   10/23/2023 updated to only process sprites that have motion
   11/04/2023 added support for enabling and disabling sprite motion
   02/24/2024 modified to use cache
   01/11/2025 updated to handle F18 text and bitmap modes
   01/22/2025 performance improvements
   02/02/2025 added f18a text 80x30 mode
   02/28/2025 removed 80x30 text mode reference
*/

#include <console.h>
#include <console_private.h>
#include <cache_private.h>
#include <conversion.h>
#include <string.h>

void console_sprite_update () {
   int i;

   // confirm display mode is standard, where sprites work, and sprite motion is enabled
   if ((cachex.console.mode == DISPLAY_MODE_STANDARD ||
        cachex.console.mode == DISPLAY_MODE_F18A_TEXT80 ||
        cachex.console.mode == DISPLAY_MODE_BITMAP) && 
       sprite_info.motion_enabled) {

      for (i = 0; i <= sprite_info.last_motion_index; i++) {               // loop through the highest sprite index where motion 
                                                                           // has been set

         if (sprite_info.sprite[i].vel_y || sprite_info.sprite[i].vel_x) { // only process where the sprite has velocity

            sprite_info.sprite[i].pos_y += sprite_info.sprite[i].vel_y;    // update the sprite's y position
            if (sprite_info.sprite[i].pos_y < SPRITE_ROW_MIN) {            // traverse from one edge of the screen to the other
               sprite_info.sprite[i].pos_y += SPRITE_ROW_MAX;
            } else if (sprite_info.sprite[i].pos_y > SPRITE_ROW_MAX) {
               sprite_info.sprite[i].pos_y -= SPRITE_ROW_MAX;
            }
      
            sprite_info.sprite[i].pos_x += sprite_info.sprite[i].vel_x;    // update the sprite's x position
            if (sprite_info.sprite[i].pos_x < SPRITE_COL_MIN) {            // traverse from one edge of the screen to the other
               sprite_info.sprite[i].pos_x += SPRITE_COL_MAX;
            } else if (sprite_info.sprite[i].pos_x > SPRITE_COL_MAX) {
               sprite_info.sprite[i].pos_x -= SPRITE_COL_MAX;
            }
   
            console_sprite_set_vdp_vals                                    // update the sprite data in vdp ram
               (i, 
                sprite_info.sprite[i].chardef, 
                sprite_info.sprite[i].color, 
                sprite_info.sprite[i].pos_y, 
                sprite_info.sprite[i].pos_x);
         }
      }
   }
}
