/* copy_with_remove.c
   This method handles the operations to implement the copy and move commands.
   change history
   06/02/2025 moved from mv.c
              added header
   06/03/2025 added verbosity for copy recursive
              adjusted the order of notes versus operations so that the notes appear after the operations execute
*/

#include <utility_private.h>
#include <batch_private.h>
#include <sys/stat.h>
#include <unistd_private.h>
#include <dylib.h>
#include <libgen.h>

int copy_with_remove (const char *program_name, bool remove_after_copy, bool copy_recursively, batch_t *batch, const char *path_source, const char *path_target, bool overwrite_files, bool interactive_overwrite, bool verbose) {

   int r = 1;

   struct stat stat_source, stat_target, stat_target_final;
   int r_source, r_target, r_target_final;
   char path_target_final[40];

   r_source = stat (path_source, &stat_source);

   if (!r_source) {
      r_target = stat (path_target, &stat_target);
      if (!r_target) {
         // both paths exist
         if (S_ISREG (stat_source.st_mode) && S_ISREG (stat_target.st_mode)) {
            // both are files
            if (overwrite_files || (interactive_overwrite && file_overwrite_query (path_target))) {
               if (paths_differ (path_source, path_target, false)) {
                  batch_file_rm (batch, path_target);
                  if (remove_after_copy && dirname_is_same (path_source, path_target)) {
                     batch_file_rename (batch, path_source, path_target);
                     batch_note (batch, path_source, path_target);
                     r = 0;
                  } else {
                     batch_file_cp (batch, path_source, path_target, stat_source.st_mode);
                     if (remove_after_copy) {
                        batch_file_rm (batch, path_source);
                     }
                     batch_note (batch, path_source, path_target);
                     r = 0;
                  }
               } else {
                  batch_note (batch, path_source, path_target);
                  r = 0;
               }
            } else {
               r = 0;
            }
         } else if (S_ISREG (stat_source.st_mode) && S_ISDIR (stat_target.st_mode)) {
            // the source is a file but the target is a directory
            dylib.strcpy (path_target_final, path_target);
            dylib.strcat (path_target_final, "/");
            dylib.strcat (path_target_final, basename ((char *) path_source));
            r_target_final = stat (path_target_final, &stat_target_final);
            if (!r_target_final) {
               if (S_ISREG (stat_target_final.st_mode)) {
                  if (overwrite_files || (interactive_overwrite && file_overwrite_query (path_target_final))) {
                     batch_file_rm (batch, path_target_final);
                     batch_file_cp (batch, path_source, path_target_final, stat_source.st_mode);
                     if (remove_after_copy) {
                        batch_file_rm (batch, path_source);
                     }
                     batch_note (batch, path_source, path_target_final);
                     r = 0;
                  } else {
                     r = 0;
                  }
               } else {
                  dylib.fputs (program_name, stderr);
                  dylib.fputs (": rename ", stderr);
                  dylib.fputs (path_source, stderr);
                  dylib.fputs (" to ", stderr);
                  dylib.fputs (path_target_final, stderr);
                  dylib.fputs (": Is a directory\n", stderr);
               }
            } else {
               batch_file_cp (batch, path_source, path_target_final, stat_source.st_mode);
               if (remove_after_copy) {
                  batch_file_rm (batch, path_source);
               }
               batch_note (batch, path_source, path_target_final);
               r = 0;
            }
         } else if (S_ISDIR (stat_source.st_mode) && S_ISDIR (stat_target.st_mode)) {
            // both are directories

            if (copy_recursively) {
               if (paths_differ (path_source, path_target, true)) {
                  dylib.strcpy (path_target_final, path_target);
                  dylib.strcat (path_target_final, "/");
                  dylib.strcat (path_target_final, basename ((char *) path_source));
                  r_target_final = stat (path_target_final, &stat_target_final);
                  if (!r_target_final) {
                     if (S_ISDIR (stat_target_final.st_mode)) {
                        if (dir_entry_count (path_target_final) == 0) {
                           dir_copy_contents_recursive (batch, path_source, path_target_final);
                           if (remove_after_copy) {
                              dir_remove_contents_recursive (batch, path_source, false);
                              batch_rmdir (batch, path_source);
                           }
                           batch_note (batch, path_source, path_target_final);
                           r = 0;
                        } else {
                           dylib.fputs (program_name, stderr);
                           dylib.fputs (": rename ", stderr);
                           dylib.fputs (path_source, stderr);
                           dylib.fputs (" to ", stderr);
                           dylib.fputs (path_target_final, stderr);
                           dylib.fputs (": Directory not empty\n", stderr);
                        }
                     } else {
                        dylib.fputs (program_name, stderr);
                        dylib.fputs (": rename ", stderr);
                        dylib.fputs (path_source, stderr);
                        dylib.fputs (" to ", stderr);
                        dylib.fputs (path_target_final, stderr);
                        dylib.fputs (": Not a directory\n", stderr);
                     }
                  } else {
                     batch_mkdir (batch, path_target_final);
                     dir_copy_contents_recursive (batch, path_source, path_target_final);
                     if (remove_after_copy) {
                        dir_remove_contents_recursive (batch, path_source, false);
                        batch_rmdir (batch, path_source);
                     }
                     batch_note (batch, path_source, path_target_final);
                     r = 0;
                  }
               } else {
                  dylib.fputs (program_name, stderr);
                  dylib.fputs (": rename ", stderr);
                  dylib.fputs (path_source, stderr); 
                  dylib.fputs (" to ", stderr);
                  dylib.fputs (path_target, stderr);
                  dylib.fputs ("/", stderr);
                  dylib.fputs (basename ((char *) path_source), stderr);
                  dylib.fputs (": Invalid argument\n", stderr);
                  r = 1;
               }
            } else {
               dylib.fputs (program_name, stderr);
               dylib.fputs (": ", stderr);
               dylib.fputs (path_source, stderr); 
               dylib.fputs (" is a directory (not copied).\n", stderr);
               r = 1;
            }

         } else if (S_ISDIR (stat_source.st_mode) && S_ISREG (stat_target.st_mode)) {
            dylib.fputs (program_name, stderr);
            dylib.fputs (": rename ", stderr);
            dylib.fputs (path_source, stderr);
            dylib.fputs (" to ", stderr);
            dylib.fputs (path_target, stderr);
            dylib.fputs (": Not a directory\n", stderr);
         }
      } else {
         if (S_ISREG (stat_source.st_mode)) {
            if (dirname_is_same (path_source, path_target)) {
               if (remove_after_copy) {
                  batch_file_rename (batch, path_source, path_target);
               } else {
                  batch_file_cp (batch, path_source, path_target, stat_source.st_mode);
               }
               batch_note (batch, path_source, path_target);
               r = 0;
            } else {
               if (dir_parent_exists (path_target)) {
                  batch_file_cp (batch, path_source, path_target, stat_source.st_mode);
                  if (remove_after_copy) {
                     batch_file_rm (batch, path_source);
                  }
                  batch_note (batch, path_source, path_target);
                  r = 0;
               } else {
                  dylib.fputs (program_name, stderr);
                  dylib.fputs (": rename ", stderr);
                  dylib.fputs (path_source, stderr);
                  dylib.fputs (" to ", stderr);
                  dylib.fputs (path_target, stderr);
                  dylib.fputs (": No such file or directory\n", stderr);
               }
            }
         } else {
            if (copy_recursively) {
               if (remove_after_copy && dirname_is_same (path_source, path_target)) {
                  batch_dir_rename (batch, path_source, path_target);
                  batch_note (batch, path_source, path_target);
                  r = 0;
               } else {
                  if (dir_parent_exists (path_target)) {
                     batch_mkdir (batch, path_target);
                     dir_copy_contents_recursive (batch, path_source, path_target);
                     if (remove_after_copy) {
                        dir_remove_contents_recursive (batch, path_source, false);
                        batch_rmdir (batch, path_source);
                     }
                     batch_note (batch, path_source, path_target);
                     r = 0;
                  } else {
                     dylib.fputs (program_name, stderr);
                     dylib.fputs (": rename ", stderr);
                     dylib.fputs (path_source, stderr);
                     dylib.fputs (" to ", stderr);
                     dylib.fputs (path_target, stderr);
                     dylib.fputs (": No such file or directory\n", stderr);
                  }
               }
            } else {
               dylib.fputs (program_name, stderr);
               dylib.fputs (": ", stderr);
               dylib.fputs (path_source, stderr); 
               dylib.fputs (" is a directory (not copied).\n", stderr);
               r = 1;
            }
         }
      }
   } else {
      dylib.fputs (program_name, stderr);
      dylib.fputs (": rename ", stderr);
      dylib.fputs (path_source, stderr);
      dylib.fputs (" to ", stderr);
      dylib.fputs (path_target, stderr);
      dylib.fputs ("/", stderr);
      dylib.fputs (basename ((char *) path_source), stderr);
      dylib.fputs (": No such file or directory\n", stderr);
   }

   return r;
}
