/* dsr_list_load.c
   This method loads the DSR list by combing through all the installed ROMs.
   change log:
   06/23/2023 initial version
   01/22/2024 added saving of the cru itself
   02/18/2024 updated to use cache-based dsr_list and initialize once
              set the root fs dsr
   12/26/2024 minor adjustments to message
   01/16/2024 removed printing load message
   06/01/2025 added display of DSRs found.
              added filtering of DSRs to disallow those with lower case characters (Classic 99 provides lower case aliases to
              uppper case DSRs.
   06/04/2025 corrected displaying so reboot with differing screen resolutions would allow the data to be written in visible manner
   06/10/2025 added IDE1 to the root options list
              added filters for DSK* map DSRs to control the list length
   06/13/2025 added configurable boot device based on location of the bootconfig file. The boot device can now be any supported file
               system type from TIPI, WDSx, IDEx or DSKx. the first existing file system of these that has the boot config file wins
   06/15/2025 added SCSI support
   07/05/2025 added more filtering of alias DSRs in support of the SCSI controller
   07/26/2025 added detection of dual R2232 cards and filtering of duplicate DSR names
   09/29/2025 substantial update to logic
   10/05/2025 modified failure display to prevent run-on into other text
*/

#include <string.h>
#include <stdio.h>
#include <stdio_private.h>
#include <cache_private.h>
#include <vdp.h>
#include <conversion.h>

//DEBUG
bool has_unix99_cfg_file (const char *dsr_name) {             // tests whether the boot config file exists
   bool b;                                                    // result

   struct PAB pab;                                            // peripheral address block
   char final_ti_path[FILENAME_MAX];                          // ti filename for the boot config file

   strcpy (final_ti_path, dsr_name);                          // calculate the full dsr path for the boot config file
   strcat (final_ti_path, ".bootconfig");

   pab.OpCode        = DSR_STATUS;                            // set the PAB values
   pab.Status        = 0x00;
   pab.VDPBuffer     = 0x0000;                                // note: this value is a bit obnoxious, but the call should not
   pab.RecordLength  = 0x00;                                  //       result in this vdp address being written to
   pab.CharCount     = 0x00;
   pab.RecordNumber  = 0x0000;
   pab.ScreenOffset  = 0x80;                                  // prime the answer as doesn't exist in case the DSR doesn't respond
   pab.NameLength    = strlen (final_ti_path);
   pab.pName         = (unsigned char *) final_ti_path;

   int r = dsrlnk (&pab, 0x3000);                             // perform the dsrlnk operation
   if (!r) {                                                  // if r is zero, the dsr call succeeded
      vdpmemread                                              // copy the result (all the PAB) back from VDP
         (0x3000, 
          (unsigned char *) &pab, 
          sizeof (pab));
      b = (pab.ScreenOffset & 0x80) != 0x80;                  // determine if the file exists. note: If the msb of the byte is high, the file doesn't exist
   } else {                                                   
      b = false;                                              // the dsrlnk failed. this should never be reached
   }

   return b;                                                  // return the result
}

void dsr_list_init () {                               
   cache.dsr_list.count = 0;
}

void dsr_write (char *text, bool is_added, unsigned int pos) {
   unsigned char v[10];
   memset (v, ' ', sizeof (v));
   memcpy (v, text, strlen (text));
   if (!is_added) {
      v[6] = '*';
   }
   v[8] = 0x00;
   vdpmemcpy (pos, v, 8);
}

bool dsr_is_uppercase (char *dsr) {
   bool is_upper_case = true;
   char *p = dsr;
   while (*p) {
      if (*p >= 'a') {
         is_upper_case = false;
         break;
      }
      p++;
   }
   return is_upper_case;
}

bool dsr_is_supported (char *dsr) {
   return !strncmp (dsr, "WDS", 3)   ||
          !strncmp (dsr, "IDE", 3)   ||
          !strncmp (dsr, "SCS", 3)   ||
          !strncmp (dsr, "DSK", 3)   ||
          !strncmp (dsr, "RS232", 5) ||
          !strncmp (dsr, "PIO", 3)   ||
          !strcmp (dsr, "TIPI")      ||
          !strcmp (dsr, "PI")        ||
          !strcmp (dsr, "TIME");
}

bool dsr_exists (char *dsr) {
   bool found = false;
   for (int i = 0; i < cache.dsr_list.count; i++) {
      found = !strcmp (dsr, cache.dsr_list.dsr[i].name);
      if (found) {
         break;
      }
   }
   return found;
}

bool dsr_is_dual_rs232 (char *dsr) {
   return dsr_exists ("RS232") &&     
          !strcmp (dsr, "RS232");
}

bool dsr_is_primary_device (char *dsr) {
   return !strcmp (dsr, "TIPI")    ||
          !strncmp (dsr, "IDE", 3) || 
          !strncmp (dsr, "SCS", 3);
}

bool dsr_is_filtered_device (char *dsr) {
   return !strncmp (dsr, "WDS", 3) ||
          !strncmp (dsr, "DSK", 3);
}

void dsr_add (char *dsr, unsigned int cru, bool first_dsr_on_cru, bool *filter_devices, unsigned int write_pos) {

   bool is_dual_rs232 = dsr_is_dual_rs232 (dsr);

   cache.dsr_list.dual_rs232 = cache.dsr_list.dual_rs232 || is_dual_rs232;

   if (first_dsr_on_cru) {
      *filter_devices = dsr_is_primary_device (dsr);
   }

   bool add_dsr = dsr_is_supported (dsr)        &&
                  !(*filter_devices && dsr_is_filtered_device (dsr)) &&
                  !dsr_exists (dsr)             &&
                  !is_dual_rs232                &&
                  (cache.dsr_list.count < FILENAME_MAX);

   if (add_dsr) {
      strcpy (cache.dsr_list.dsr[cache.dsr_list.count].name, dsr);
      cache.dsr_list.dsr[cache.dsr_list.count].cru = cru;
      cache.dsr_list.count++;
   }

   dsr_write (dsr, add_dsr, write_pos);
}

void dsr_list_load () {

   bool filter_devices = false;

   unsigned int pos = 0x0060;                                 // initialize the screen write position

   int len;
   struct DeviceRomHeader* dsrrom = (struct DeviceRomHeader*) 0x4000;;
   struct NameLink* dsrlinks;

   dsr_list_init ();                                         // initialize the dsr list

   dsr_write ("DSRs:", true, pos);                           // write title for DSRs
   pos += 8;

   char dsr_name[10];
   bool is_first_dsr_on_cru;

   // start scanning crus at the first possible cru address
   int cruscan = 0x1000;
   while(cruscan < 0x2000) {
      enable_rom (cruscan);
      if (dsrrom->flag == 0xAA) {
 
         dsrlinks = dsrrom->dsrlnk;

         is_first_dsr_on_cru = true;

         while(dsrlinks != 0) {

            // add the dsr to the list
            // capture the length - first char of the name is the length
            len = (int) dsrlinks->name[0];

            memcpy (dsr_name, &dsrlinks->name[1], len);    // capture the name - second char is the start of the string
            dsr_name[len] = 0x00;                          // null terminate the C string

            dsr_add (dsr_name, cruscan, is_first_dsr_on_cru, &filter_devices, pos);
            pos += 8;

            is_first_dsr_on_cru = false;

            dsrlinks = dsrlinks->next;
         }
      }
 
      disable_rom (cruscan);
      cruscan += 0x0100;

      // break out if the DSR list is full
      if (cache.dsr_list.count == FILENAME_MAX) {
         break;
      }
   }

   // capture the root dsr

   const char *dsr[] = {"TIPI", "WDS", "IDE", "SCS", "DSK"};                   // candidate root directory devices. TIPI, WDSx,
   const int npaths = 5;                                                       // IDEx, or DSKx will be considered

   int i, j;

   vdpmemcpy (0x0030, (unsigned char *) "Boot dev: ", 10);                     // write title

   strcpy (cache.dsr_list.root_dsr, "");
   for (i = 0; i < npaths; i++) {
      for (j = 0; j < cache.dsr_list.count; j++) {
         if (!strncmp (dsr[i], cache.dsr_list.dsr[j].name, strlen (dsr[i])) && // match dsr[i] with dsr[i] length &&
            strlen (cache.dsr_list.dsr[j].name) == 4 &&                        // the actual DSR len is 4, by TI conventions &&
            has_unix99_cfg_file (cache.dsr_list.dsr[j].name)) {                // and the boot config exists there
                                                                               // note: intentionally ignoring named volumes...
            strcat (cache.dsr_list.root_dsr, "/");                             // generate the root directory path
            strcat (cache.dsr_list.root_dsr, cache.dsr_list.dsr[j].name);

            vdpmemcpy                                                          // write the name of the found DSR
               (0x003a, 
                (unsigned char *) cache.dsr_list.dsr[j].name, 
                strlen (cache.dsr_list.dsr[j].name));

            break;                                                             // exit the inner for loop
         }
      }
      if (strlen (cache.dsr_list.root_dsr)) {                                  // exit the outer for loop if the root dir is found
         break;
      }
   }

   if (!strlen (cache.dsr_list.root_dsr)) {                                    // if there is no root dir, cannot boot
      vdpmemcpy (0x003a, (unsigned char *) "none", 4);                         // show the boot device
      vdpmemcpy (0x0013, (unsigned char *) "fail", 4);                         // indicate boot failure
      while (1);
   }
}
