/* exec_uea5.c
   This method executes an Unix EA5 image first by loading it and then branching to it
   change log:
   02/07/2024 initial version, with heavy borrowing from exec_ea5
   02/11/2024 corrected last spinner from underscore to backslash
   02/12/2024 added test to confirm the file is executable
              added confirmation that the correct number of bytes were read
   02/14/2024 updated to read and write spinner with console methods
   02/18/2024 removed use of static variable data
   02/24/2024 migrated spinner functionality to spinner package
              broad changes for cache and dylib
   03/06/2024 removed exit call triggered by failure--allow return from method
   01/04/2025 added SAMS caching of programs
   01/04/2025 modified code to relatively easily disable if necessary
   01/06/2025 minor updates
   07/05/2025 added saving the program mapped pages
   07/09/2025 corrected page vs region
   07/14/2025 added unlocking of the previous program's region and locking of the new program's region
*/

#include <vdp.h>
#include <string.h>
#include <stdio_private.h>
#include <unistd_private.h>
#include <conversion.h>
#include <console.h>
#include <stdlib.h>
#include <sams_private.h>
#include <cache_private.h>

static void exec_uea5_launch (unsigned int faddr) {
   GPLWSR11 = faddr;
   __asm__("b @exec_prg");
}

int exec_uea5 (const char *path) {

#define FLAG  0
#define BSIZE 1
#define ADDR  2

   unsigned int metadata[3];                                   // values in order: FLAG, BSIZE, ADDR
   unsigned int maddr;                                         // memory address which to load executable image
   int rsize;                                                  // desired number of bytes to read
   int n;                                                      // number of bytes actually received
   int tbytes = 0;                                             // total number of executable bytes read
   int r      = UNDEFINED;                                     // the return value

   sams_unlock_region (cache.sams.region_in_use);                 // unlock the previous program's region so it can later be swapped out if needed for other uses
   r = sams_find_cache (path);                                    // find the cached program, or set the memory to a free/usable block
                                                                  // return 0 = found, 1 = not found, load the file
   if (r) {                                                       // determine if the file should be processed or not

      FILE *f = fopen (path, "rb");                               // open the file

      if (f) {                                                    // confirm the file was opened successfully
         if (file_handle_is_executable (f)) {                     // confirm the file is executable
            rsize  = FILE_BINARY_BLOCK_SIZE;                      // align to the block size
            n      = fread (&metadata, 1, sizeof (metadata), f);  // read the executable's header metadata
            rsize -= n;                                           // reduce the next read size by the bytes already written to ensure

                                                                  // subsequent reads fall on sector boundaries, thus eliminating multiple
                                                                  // reads of the same sector
            maddr   = metadata[ADDR];                             // set the initial load address
            tbytes += n;
   
            while (n) {                                           // read in the entire uea5 file
               n       = fread ((void *) maddr, 1, rsize, f);     // read the next set of data
               tbytes += n;                                       // count the number of bytes read
               maddr  += n;                                       // update the write addr based on the number of bytes read
               rsize   = FILE_BINARY_BLOCK_SIZE;                  // set rsize to the standard block size. this simply realigns the reading
                                                                  // with the sectors. technically this is only needed once
            }
   
            if (tbytes == metadata[BSIZE]) {                      // confirm number of bytes read equals the number of bytes specified in the header
               r = 0;                                             // set success
            }
         }

         fclose (f);                                              // close the file

      }
   } else {
      metadata[ADDR] = 0xa000;                                    // a little klundy, but in this implementation all programs have this entry point
   }

   sams_save_proc_pages ();                                    // save this program's process pages
   sams_lock_region (cache.sams.region_in_use);                // lock the process' region so it can't be swapped out while running

   if (r == 0) {                                               // confirm success
      exec_uea5_launch (metadata[ADDR]);                       // finally launch the uea5 executable
   } else {
      proc_cache_init ();                                      // kill off the future processes since this one failed. could be more graceful, but...
   }

   return r;                                                   // return the result
}
