/* fclose.c
   This method closes a currently open file stream.
   change log:
   06/23/2023 initial version
   08/14/2023 updated to always set ftype to undefined when its a valid file
   09/02/2023 disabled PIO
   11/23/2023 updated to use separate dsr
   12/20/2023 added binary file support
   12/22/2023 updated to capture an error on closing binary files
   02/07/2024 modified to use common method file_write_binary_file_status
              added comment that this method includes some duplicative code with dylib.fputs
   02/16/2024 added write cache handing
   02/24/2024 added spinner
*/

#include <constants.h>
#include <stdio.h>
#include <stdio_private.h>
#include <spinner_private.h>
#include <vdp.h>

int fclose (FILE *f) {
   int r;


   if (f) {
      switch (f->ftype) {
         case FTYPE_CONSOLE_IN:
         case FTYPE_CONSOLE_OUT:
//       case FTYPE_PARALLEL_IN:
         case FTYPE_PARALLEL_OUT:
            r = 0;
            break;
         case FTYPE_REG_FILE:
            r = 0;

            spinner_read ();

            if (f->is_binary) {

               if (f->dsr->read_write_op == DSR_WRITE) {
                  if (f->wc_active) {

                     spinner_write ();

                     f->dsr->pab.OpCode    = DSR_WRITE;
                     f->dsr->pab.CharCount = FILE_BINARY_BLOCK_SIZE;
                     r += dsrlnk (&f->dsr->pab, f->dsr->vdp_pab_buffer_addr);
                     f->wc_active = false;
                  }

                  r += file_write_binary_file_status (f);
               }

            } else {

               // write any remaining data that has been buffered
               if (f->dsr->vdp_data_buffer_wp != f->dsr->vdp_data_buffer_addr) {

                  spinner_write ();

                  // note that this code is duplicative of some in dylib.fputs, but fputs is a fairly substantial size so removing this
                  // code will bloat executables that do not use dylib.fputs
                  f->dsr->pab.OpCode    = DSR_WRITE;
                  f->dsr->pab.CharCount = f->dsr->vdp_data_buffer_wp - f->dsr->vdp_data_buffer_addr;
                  r += dsrlnk (&f->dsr->pab, f->dsr->vdp_pab_buffer_addr);
               }
            }

            // close the file

            spinner_write ();

            f->dsr->pab.OpCode = DSR_CLOSE;
            r += dsrlnk (&f->dsr->pab, f->dsr->vdp_pab_buffer_addr);
            if (r) {
               if (f->dsr->read_write_op == DSR_READ) {
                  // file read at the end of the file gets upset, so just ignore this
                  r = 0;
               }
            }

            f->dsr->in_use = false;

            spinner_restore ();

            break;
         default:
            r = -1;
            break;
      }
      f->ftype  = UNDEFINED;
      f->in_use = false;
   } else {
      r = -1;
   }

   
   return r;
}
