/* fgets.c
   This method performs the file get string function.
   change log:
   03/06/2024 new version integrating use of binary files with text
   06/08/2025 added writing of \n on console input via fgets
   06/09/2025 handle EOF from console
   06/11/2025 removed rs232.h include
*/ 

#include <stdio.h>
#include <string.h>
#include <assert.h>
#include <stdio_private.h>
#include <longdivmod.h>
#include <math.h>
#include <console.h>
#include <conversion.h>
#include <cache_private.h>
#include <dylib.h>
#include <vdp.h>
#include <errno.h>

char *fgets (char * str, int size, FILE * stream) {

   char *str_return = str;
   int str_bytes;
   long file_pos;
   long str_used;
   int bytes_to_read;
   int block_bytes_available;
   int bytes_read;
   int i;
   int r;

   switch (stream->ftype) {

      case FTYPE_REG_FILE:               // handle regular file

         if (stream->is_binary) {

            str_bytes     = size - 1;
            file_pos      = ftell (stream);
            str_used      = 0;
            bytes_to_read = 1;
         
            while (bytes_to_read) {
               block_bytes_available = FILE_BINARY_BLOCK_SIZE - __moddi3 (stream->bstate.pos, FILE_BINARY_BLOCK_SIZE);
               bytes_to_read         = min (str_bytes, block_bytes_available);
               bytes_read            = fread (str, 1, bytes_to_read, stream);
               if (bytes_read > 0) {
                  for (i = 0; i < bytes_read; i++) {
                     str_bytes--;
                     str_used++;
                     if (*str == '\n') {
                        str++;
                        bytes_to_read = 0;
                        break;
                     }
                     str++;
                  }
               } else {
                  bytes_to_read = 0;
               }
            }
            *str = 0x00;
   
            file_pos += str_used;
            fseek (stream, file_pos, SEEK_SET);
   
            if (!str_used) {
               str_return = NULL;
            }

         } else {

            spinner_read ();

            spinner_write ();

            stream->dsr->pab.OpCode    = DSR_READ;                       // set the DSR op code
            stream->dsr->pab.CharCount = 0xff;                           // default the count
            r = dsrlnk (&stream->dsr->pab, stream->dsr->vdp_pab_buffer_addr); // call the TI DSR

            vdpmemread (stream->dsr->vdp_pab_buffer_addr, (unsigned char *) &stream->dsr->pab, 9);

            if (!r) {                                                    // test for read success
               char *p = str;                                            // set the write pointer
               if (stream->dsr->pab.CharCount != 255) {                  // test against 255 which is an invalid count
                  VDP_SET_ADDRESS (stream->dsr->vdp_data_buffer_addr);   // set the read pointer
                  for (int i = 0; i < stream->dsr->pab.CharCount; i++) { // copy from vdp to the string buffer
                     *p = VDPRD;
                     p++;
                  }
               }
               *p = '\n';                                                // set last readable character to a LF
               p++;                                                      // advance one more position
               *p = 0x00;                                                // null terminate
            } else {
               errno       = r;                                          // set errno to this error value
               stream->eof = true;                                       // set end of file
               str_return  = NULL;                                       // set the return value
            }

            spinner_restore ();

         }

         break;

      case FTYPE_CONSOLE_IN:                                     // handle a console-based file

         if (dylib.console_gets (str, size, false)) {            // redirect to the console gets
            strcat (str, "\n");                                  // input received since console_gets didn't return null
         } else {
            stream->eof = true;                                  // at EOF (user entered ctrl-D)
            str_return  = NULL;                                  // set return to NULL
         }

         break;

      default:                                                   // catch others which are errors

         str_return = NULL;                                      // set the return value

         break;
   }

   return str_return;
}
