/* file_get_dsr_status.c
   This method returns the DSR status of a file. 
   The return should first be tested for UNDEFINED, and if not that then assessed according to the following:
   MSB. Source: File Management Specification for the TI-99/4 Home Computer, Table 3-1
   Bit 0 - if set the file doesn't exist
   Bit 1 - if set the file is protected
   Bit 2 - reserved for future (hah, never, since written around 1980) use
   Bit 3 - data type. if set the type is binary/internal, otherwise ascii/display
   Bit 4 - if set the file is a program type
   Bit 5 - record type. if set, the record type is variable, otherwise fixed
   Bit 6 - physical end of file. If set this means the disk is full
   Bit 7 - logical end of file. The file is at the end of the file and can be written to, but a read would fail
   change log:
   07/01/2023 initial version
   07/22/2023 added test for null file name, which causes the call to dsrlnk to get upset and write all over vdp memory (yikes)
   07/23/2023 added reference to the TI specification document
   08/14/2023 updated for unix paths
   11/21/2023 updated to use FILENAME_MAX
   11/23/2023 updated to use separate dsr
   03/01/2024 update to use dylib
   01/10/2025 updated to use dylib.file_get_available
   01/22/2025 performance improvements
*/

#include <vdp.h>
#include <string.h>
#include <constants.h>
#include <stdio.h>
#include <stdio_private.h>
#include <unistd.h>
#include <unistd_private.h>
#include <dylib.h>

int file_get_dsr_status (const char *name) {
   int r = UNDEFINED;

   FILE *f = dylib.file_get_available (true);
   if (f && dylib.strlen (name)) {

      char final_ti_path[FILENAME_MAX];
      dylib.generate_final_path (final_ti_path, (char *) name, false);

      f->ftype = FTYPE_REG_FILE;
      file_init_pab (f);       

      // set the standard open file values
      f->dsr->pab.OpCode     = DSR_STATUS;
      f->dsr->pab.NameLength = dylib.strlen (final_ti_path);
      f->dsr->pab.pName      = (unsigned char *) final_ti_path;

      // perform the dsrlink operation
      r = dylib.dsrlnk (&f->dsr->pab, f->dsr->vdp_pab_buffer_addr);
      if (!r) {
         vdpmemread (f->dsr->vdp_pab_buffer_addr, (unsigned char *) &f->dsr->pab, 9);
         r = f->dsr->pab.ScreenOffset;
      }
   }

   return r;
}
