/* fopen_regular_file.c
   These methods open a file and provide support for DSRs.
   change log:
   06/23/2023 initial version
   08/14/2023 updated for unix paths
   08/25/2023 removed support for parallel port input. Left code commented out. fgets doesn't support parallel input either
   10/21/2023 removed window references
   11/21/2023 updated to use FILENAME_MAX
   11/22/2023 added setting eof
   11/23/2023 updated to use separate dsr
   12/19/2023 added binary file support
   12/22/2023 updated to test dsr operation results associated with binary files
   02/07/2024 modified to use common method file_write_binary_file_status
   02/16/2024 added write cache handling
   02/24/2024 added spinner
   03/06/2024 moved these contents from fopen
              added set_file_info to handle common code in the main method
              cleanup
   03/07/2024 corrected to set truncation on binary write open
   12/21/2024 cleanup in preparation for testing files for existance
   12/25/2024 added testing for files whose modes require existance
   01/04/2025 wrapped testing of files for existance in def / ifdef statements. Awaiting a fix to TIPI from jedimatt42 that will
              perform the checks there
   06/02/2025 modified to read the actual text file record length after file open. Necessary fix for Classic99
   06/24/2025 renamed file_is_valid_path to file_has_valid_dsr_name
   07/05/2025 augmented DSR path tests
   10/13/2025 updated ref from file_has_valid_dsr_name to file_get_dsr_cru
*/

#include <string.h>
#include <constants.h>
#include <stdio.h>
#include <stdio_private.h>
#include <console.h>
#include <errno.h>
#include <unistd_private.h>
#include <spinner_private.h>
#include <vdp.h>
#include <stddef.h>
#include <stdlib.h>

// #define TEST_FILE_EXISTS

void set_file_info (FILE *f, unsigned char opcode, const char *name, bool is_binary, unsigned int read_write_op, unsigned char pab_status, unsigned char rec_len) {

   f->ftype                   = FTYPE_REG_FILE;                         // this is a regular file, although it can still be redirected I/O
   f->eof                     = false;                                  // not at end of file
   f->is_binary               = is_binary;
   f->dsr->read_write_op      = read_write_op;
   f->dsr->vdp_data_buffer_wp = f->dsr->vdp_data_buffer_addr;           // reset the write position 

   file_init_pab (f);                                                   // initialize the pab
   f->dsr->pab.OpCode         = opcode; 
   strcpy (f->dsr->pab_filename, name);
   f->dsr->pab.NameLength     = strlen (f->dsr->pab_filename);          // set the filename length
   f->dsr->pab.pName          = (unsigned char *) f->dsr->pab_filename; // set the filename
   f->dsr->pab.Status         = pab_status;
   f->dsr->pab.RecordLength   = rec_len;
   f->dsr->pab.RecordNumber   = 0;                                      // set to the first record
}

#ifdef TEST_FILE_EXISTS

int file_exists (FILE *f, const char *name) {
   set_file_info (f, DSR_STATUS, name, false, 0, 0, 0);
 
   int r = !dsrlnk (&f->dsr->pab, f->dsr->vdp_pab_buffer_addr);
   if (r) {
      vdpmemread (f->dsr->vdp_pab_buffer_addr, (unsigned char *) &f->dsr->pab, 9);
      r = !(f->dsr->pab.ScreenOffset & 0x01);
   }
   return r;
}

#endif

FILE *fopen_regular_file (const char *name, const char *mode) {
   FILE *f = NULL;

   spinner_read ();

   // confirm first that the filename includes a valid named DSR. Without that no DSR will be called nor will an error
   // be set on the DSR call itself, which then appears as if the file was opened successfully.

   char final_ti_path[FILENAME_MAX];
   generate_final_path (final_ti_path, (char *) name, false);

// TEMP FIX, added this line and commented out the following lines
   f = file_get_available (true);       // get an available file index
   if (!f) {
      return NULL;
   }
/*
   if (file_get_dsr_cru (final_ti_path)) { // verify this path has a valid cru
      f = file_get_available (true);       // get an available file index
      if (!f) {
         return NULL;
      }
   }
*/

   bool binary_trunc_file = false;

   // capture the mode in the PAB status field, and retain the type of read/write operation for later use
   if (!strcmp (mode, "r")) {        // text - read only
#ifdef TEST_FILE_EXISTS
      if (file_exists (f, final_ti_path)) {
#endif
         set_file_info (f, DSR_OPEN, final_ti_path, false, DSR_READ, DSR_TYPE_DISPLAY | DSR_TYPE_VARIABLE | DSR_TYPE_SEQUENTIAL | DSR_TYPE_INPUT, 0);
#ifdef TEST_FILE_EXISTS
      } else {
         return NULL;
      }
#endif
   } else if (!strcmp (mode, "w")) { // text - write only
      set_file_info (f, DSR_OPEN, final_ti_path, false, DSR_WRITE, DSR_TYPE_DISPLAY | DSR_TYPE_VARIABLE | DSR_TYPE_SEQUENTIAL | DSR_TYPE_OUTPUT, 0);
   } else if (!strcmp (mode, "a")) { // text - append
      set_file_info (f, DSR_OPEN, final_ti_path, false, DSR_WRITE, DSR_TYPE_DISPLAY | DSR_TYPE_VARIABLE | DSR_TYPE_SEQUENTIAL | DSR_TYPE_APPEND, 0);
   } else if (!strcmp (mode, "rb+")) { // binary reading and writing, but the file must exist
#ifdef TEST_FILE_EXISTS
      if (file_exists (f, final_ti_path)) {
#endif
         set_file_info (f, DSR_OPEN, final_ti_path, true, DSR_WRITE, DSR_TYPE_INTERNAL | DSR_TYPE_FIXED | DSR_TYPE_RELATIVE | DSR_TYPE_UPDATE, FILE_BINARY_BLOCK_SIZE);
#ifdef TEST_FILE_EXISTS
      } else {
         return NULL;
      }
#endif
   } else if (!strcmp (mode, "wb") || !strcmp (mode, "wb+")) { // binary writing, truncating if the file exists
      set_file_info (f, DSR_OPEN, final_ti_path, true, DSR_WRITE, DSR_TYPE_INTERNAL | DSR_TYPE_FIXED | DSR_TYPE_RELATIVE | DSR_TYPE_UPDATE, FILE_BINARY_BLOCK_SIZE);
      binary_trunc_file = true;
   } else if (!strcmp (mode, "rb")) {  // binary reading
#ifdef TEST_FILE_EXISTS
      if (file_exists (f, final_ti_path)) {
#endif
         set_file_info (f, DSR_OPEN, final_ti_path, true, DSR_READ, DSR_TYPE_INTERNAL | DSR_TYPE_FIXED | DSR_TYPE_RELATIVE | DSR_TYPE_INPUT, FILE_BINARY_BLOCK_SIZE);
#ifdef TEST_FILE_EXISTS
      } else {
         return NULL;
      }
#endif
   } else {
      // invalid mode was provided
      return NULL;
   }

   spinner_write ();

   // perform the dsrlink operation
   int r = dsrlnk (&f->dsr->pab, f->dsr->vdp_pab_buffer_addr);
   if (!r) {
      // read the record length
      vdpmemread (f->dsr->vdp_pab_buffer_addr + OFFSETOF (struct PAB, RecordLength), &f->dsr->pab.RecordLength, 1);
   }

   // assess the return value, 0 is OK, non-zero is an error
   if (!r && f->is_binary) {

      // determine if this file should be truncated
      if (binary_trunc_file) {

         // set initial values
         f->bstate.pos        = 0;
         f->bstate.len        = 0;
         f->bstate.last_block = 0;

         r = file_write_binary_file_status (f);
      } else {

//       spinner_write ();

         // this must be an existing file, capture the stats from the first record on disk
         f->dsr->pab.OpCode       = DSR_READ;
         f->dsr->pab.RecordNumber = 0;
         f->dsr->pab.CharCount    = FILE_BINARY_BLOCK_SIZE;
         r = dsrlnk (&f->dsr->pab, f->dsr->vdp_pab_buffer_addr);

         // copy the data from vdp to ram
         vdpmemread (f->dsr->vdp_data_buffer_addr, (unsigned char *) &f->bstate, sizeof (binary_state_t));

         // initialize the file position pointer
         f->bstate.pos = 0;

      }

      // initialize the write cache active flag
      f->wc_active = false;
   }

   if (!r) {
      // all is well, set the dsr and file as in use
      f->dsr->in_use = true;
      f->in_use      = true;
   } else {
      // an error occurred, set errno
      errno = r;
      f     = NULL;
   }

   spinner_restore ();

   return f;
}
