/* fputs.c
   This method implements file put string.
   change log:
   03/07/2024 new method that supports text files using binary interfaces
   01/10/2025 updated to use dylib.fwrite
   01/20/2025 moved to kernel, thus removed most dylib calls
              moved console_puts to dylib call
   12/12/2025 modified to handle printing to alternative ports
*/

#include <stdio.h>
#include <string.h>
#include <assert.h>
#include <stdio_private.h>
#include <longdivmod.h>
#include <math.h>
#include <console.h>
#include <rs232.h>
#include <conversion.h>
#include <cache_private.h>
#include <dylib.h>
#include <vdp.h>
#include <errno.h>

int fputs (const char *restrict s, FILE *restrict stream) {

   int bytes_out;
   int bytes_written;
   char *p;
   unsigned int cru;

   int r = 0;

   switch (stream->ftype) {                                   // switch on the file type

      case FTYPE_REG_FILE:                                    // handle regular file

         if (stream->is_binary) {

            bytes_out     = strlen (s);                       // get the number of bytes to be written
            bytes_written = fwrite (s, 1, bytes_out, stream); // write the bytes to the file
   
            if (bytes_out != bytes_written) {                 // confirm the number of bytes written is the same as requested
               r = -1;                                        // not so, so set an error condition
            }

         } else {

            spinner_read ();

            int len       = strlen (s);                       // determine the length of the string
            p             = (char *) s;                       // capture a pointer to the string
            bool set_addr = true;                             // flag that the vdp address needs to be set
            for (int i = 0; i < len; i++) {                   // loop through all the string
               if (set_addr) {
                  VDP_SET_ADDRESS_WRITE (stream->dsr->vdp_data_buffer_wp); // write the address to be written to the vdp write
                                                                           // addr port
                  set_addr = false;                           // reset the write flag
               }
   
               switch (*p) {
                  case '\n':                                  // handle CR/LF
                  case '\r':
   
                     spinner_write ();
   
                     p++;                                     // advance beyond the CR or LF
                     // write the file
                     stream->dsr->pab.OpCode         = DSR_WRITE;
                     stream->dsr->pab.CharCount      = stream->dsr->vdp_data_buffer_wp - stream->dsr->vdp_data_buffer_addr;
                     r -= dsrlnk (&stream->dsr->pab, stream->dsr->vdp_pab_buffer_addr);
                     stream->dsr->vdp_data_buffer_wp = stream->dsr->vdp_data_buffer_addr;
                     set_addr = true;                         // set to write the vdp address for subsequent update
                     break;
                  default:
                     VDPWD = *p;                              // not a carriage return so just write the next byte to vdp
                     p++;                                     // move to the next position
                     stream->dsr->vdp_data_buffer_wp++;       // update the vdp data buffer write position
                     break;
               }
            }

            spinner_restore ();

         }

         break;

      case FTYPE_CONSOLE_OUT:                              // handle writing to the console

         dylib.console_puts (s);                           // just write using the console routine

         break;

      case FTYPE_PARALLEL_OUT:                             // handle output to the parallel port

         // same program from https://www.unige.ch/medecine/nouspikel/ti99/rs232c.htm#CRU%20map
   
         if (stream->fd) {         // positive is the 2nd card
            cru = RS232_ALT_CARD;
         } else {                  // zero is the first card
            cru = RS232_CARD;
         }

         enable_rom (cru);                                 // enable the card and its DSR

         __asm__("sbz 1");                                 // set the parallel port for writing
         p = (char *) s;                                   // capture a pointer to the string to be sent
         while (*p != 0x00) {                              // loop through all the string
            __asm__("sbo 2");                              // kludge: force the handshake high
            PIO_PORT = *p;                                 // write the byte
            __asm__("sbz 2");                              // set the handshake low to indicate the byte was sent
            p++;
         }

         disable_rom (cru);                                // disable the card

         break;

      default:

         r = -1;                                           // not a valid file type

         break;
   }

   return r;                                               // return the result
}
