/* fread.c
   This method reads data from a file stream. Generally this method should only be used with binary files. 
   change log:
   12/20/2023 initial version
   12/22/2023 ensure that reads cannot occur past end of file
   12/29/2023 modified to call the correct div/mod methods while the compiler is being corrected
              long __divdi3 (long numerator, long denominator);
              long __moddi3 (long numerator, long denominator);
   02/15/2024 added read cache handling
   02/16/2024 added write cache handling
   02/24/2024 added spinner
*/

#include <stdio.h>
#include <stdio_private.h>
#include <math.h>
#include <vdp.h>
#include <conversion.h>
#include <longdivmod.h>

#include <spinner_private.h>

int fread (void *restrict ptr, int size, int nitems, FILE *f) {

   int r = 0;

   spinner_read ();

   // ensure this is a binary file
   if (f->is_binary) {

      int total_bytes, bytes_remaining, block_num, block_pos, block_bytes;
      char *p = (char *) ptr;


      // calculate the total number of bytes to be written:
      // nominally the number of bytes to be read is size * nitems, but that may be truncated by overrunning the end of the file;
      // the total number of bytes calculated can be negative if fseek was used to move beyond the end of file (a legal operation),
      // so ensure the final tally is non-negative.
      total_bytes = max (min (size * nitems, f->bstate.len - f->bstate.pos), 0);

      // loop until all bytes are read
      bytes_remaining = total_bytes;
      while (bytes_remaining) {

         // calculate where to read
         block_num   = __divdi3 (f->bstate.pos, FILE_BINARY_BLOCK_SIZE) + 1; // add one since the first block contains the binary file stats
         block_pos   = __moddi3 (f->bstate.pos, FILE_BINARY_BLOCK_SIZE);
         block_bytes = min (FILE_BINARY_BLOCK_SIZE - block_pos, bytes_remaining);

         if (f->dsr->pab.RecordNumber != block_num) {

            if (f->wc_active) {
               spinner_write ();

               // read the data from storage
               f->dsr->pab.OpCode       = DSR_WRITE;
               f->dsr->pab.CharCount    = FILE_BINARY_BLOCK_SIZE;

               r = dsrlnk (&f->dsr->pab, f->dsr->vdp_pab_buffer_addr);
               if (r) {
                  break;
               }
               f->wc_active = false;
            }

            spinner_write ();

            // read the data from storage
            f->dsr->pab.OpCode       = DSR_READ;
            f->dsr->pab.RecordNumber = block_num;
            f->dsr->pab.CharCount    = FILE_BINARY_BLOCK_SIZE;
            r = dsrlnk (&f->dsr->pab, f->dsr->vdp_pab_buffer_addr);


            if (r) {
               break;
            }
         }

         // copy the data from vdp to the target object
         vdpmemread (f->dsr->vdp_data_buffer_addr + block_pos, (unsigned char *) p, block_bytes);
         p += block_bytes;

         // update stats
         f->bstate.pos   += block_bytes;
         bytes_remaining -= block_bytes;
      }

      // set the return value
      if (!r) {
         r = total_bytes / size;
      }

   } 

   spinner_restore ();

   return r;
}
