/* fwrite.c
   This method writes to a file stream. Generally this method should only be used with binary files. 

   Unfortunately some GCC optimizations will replace calls for one method with another. In particular fputs ("\n", stdout) 
   is replaced with an equivalent call to fwrite! To address this unwelcome behavior, the data to be written is collected 
   and then written with a single call to fputs, which was likely the intended method to be called.
   change log:
   06/23/2023 initial version
   12/10/2023 changed refernce to math.h
   12/19/2023 added binary file support
   12/22/2023 added binary file error handling
   12/29/2023 added hack to call the long div/mod methods while the compiler is being fixed
              long __divdi3 (long numerator, long denominator);
              long __moddi3 (long numerator, long denominator);
   01/03/2024 misc cleanup after debugging
   02/15/2024 added read cache handling
   02/16/2024 added write cache handing
   02/24/2024 added spinner
   03/07/2024 modified to use dylib.dsrlnk
   01/10/2025 modified to call dylib.fputs rather than fputs so fputs didn't get built into the 8 KB ROM where it doesn't fit
   01/20/2025 removed dylib references in appropriate for kernel
   01/20/2025 replaced dylib.fputs with fputs
*/

#include <stdio.h>
#include <stdio_private.h>
#include <math.h>
#include <vdp.h>
#include <longdivmod.h>
#include <conversion.h>
#include <spinner_private.h>
#include <dylib.h>

int fwrite (const void *restrict ptr, int size, int nitems, FILE * restrict f) {

   int r = 0;

   // determine if this is a binary or text file that is open
   if (f->is_binary) {

      spinner_read ();


      int bytes_remaining, block_num, block_pos, block_bytes;

      // capture the initial address of data to be written
      char *p = (char *) ptr;

      // calculate the total number of bytes to be written
      bytes_remaining = size * nitems;

      // loop until all bytes are written
      while (bytes_remaining) {

         // calculate where to start writing
         block_num   = __divdi3 (f->bstate.pos, FILE_BINARY_BLOCK_SIZE) + 1; // add one since the first block contains the binary file stats
         block_pos   = __moddi3 (f->bstate.pos, FILE_BINARY_BLOCK_SIZE);
         block_bytes = min (FILE_BINARY_BLOCK_SIZE - block_pos, bytes_remaining);

         if (f->dsr->pab.RecordNumber != block_num && f->wc_active) {

            spinner_write ();

            f->dsr->pab.OpCode    = DSR_WRITE;
            f->dsr->pab.CharCount = FILE_BINARY_BLOCK_SIZE;
            r = dsrlnk (&f->dsr->pab, f->dsr->vdp_pab_buffer_addr);
            f->wc_active = false;
            if (r) {
               r = UNDEFINED;
                  break; // breaks out of the for loop, need one more break to get out of the while loop (a few lines later)
               }
         }

         // handle creation of blocks after a seek operation has moved beyond the current physical end of the file. This will write blank
         // blocks between the span. The TI file system doesn't support sparse files so this will indeed use space on the storage device
         if (f->bstate.last_block + 1 < block_num) {
            vdpmemset (f->dsr->vdp_data_buffer_addr, 0x00, FILE_BINARY_BLOCK_SIZE);
            f->dsr->pab.OpCode    = DSR_WRITE;
            for (int eb = f->bstate.last_block + 1; eb < block_num; eb++) {
               spinner_write ();

               f->dsr->pab.RecordNumber = eb;
               f->dsr->pab.CharCount    = FILE_BINARY_BLOCK_SIZE;
               r = dsrlnk (&f->dsr->pab, f->dsr->vdp_pab_buffer_addr);
               if (r) {
                  r = UNDEFINED;
                  break; // breaks out of the for loop, need one more break to get out of the while loop (a few lines later)
               }
               f->bstate.last_block++;
            }
            if (r) {
               break;    // break out of the while loop now since we needed a double break from within the for loop
            }
         }

         // load the existing block if the entire block isn't being overwritten
         if (block_bytes != FILE_BINARY_BLOCK_SIZE && block_num <= f->bstate.last_block) {
            if (f->dsr->pab.RecordNumber != block_num) {

               spinner_write ();

               f->dsr->pab.OpCode       = DSR_READ;
               f->dsr->pab.RecordNumber = block_num;
               f->dsr->pab.CharCount    = FILE_BINARY_BLOCK_SIZE;
               r = dsrlnk (&f->dsr->pab, f->dsr->vdp_pab_buffer_addr);
               if (r) {
                  r = UNDEFINED;
                  break;
               }
               f->bstate.last_block = block_num;
            }
         } else {
            // this will be a new block so initialize it unless the entire block is going to be written
            if (block_bytes < FILE_BINARY_BLOCK_SIZE) {
               vdpmemset (f->dsr->vdp_data_buffer_addr, 0x00, FILE_BINARY_BLOCK_SIZE);
            }
         }

         // write the data to the vdp
         vdpmemcpy (f->dsr->vdp_data_buffer_addr + block_pos, (unsigned char *) p, block_bytes);
         p += block_bytes;

         // cache the block, waiting to later write to storage
         f->dsr->pab.RecordNumber = block_num;
         f->wc_active = true;

         // update values
         f->bstate.pos       += block_bytes;
         f->bstate.len        = max (f->bstate.len, f->bstate.pos);
         f->bstate.last_block = max (f->bstate.last_block, block_num);
         bytes_remaining     -= block_bytes;
      }

      // set the return value
      if (r != UNDEFINED) {
         r = nitems;
      }

      spinner_restore ();

   } else {

      // process text
      char s[258];
      char *p = (char *) ptr;
      int i, j;
      for (i = 0; i < nitems; i++) {
         for (j = 0; j < size; j++) {
            s[j] = *p;
            p++;
         }
         s[size] = 0x00;
         r = min (r, fputs (s, f));
      }
   }

   return r;
}
