/* generate_unix_path.c
   This method generates full paths given the current working directory and either absolute or relative paths to
   directories and files.  Path elements in Unix are separated solely by forwared slashes (/). 

   Example paths are:
   - /WDS1/lib/speech      directory
   - /DSK1                 volume
   - /DSK2                 volume
   - /DSK2/GREP            EA5 program

   Current working directories are always valid paths. Requested paths are either absolute, describing the full
   path to a volume, directory or file, or a relative path.

   Generally the final path is current working path + requested path, as processed. The .. and . directories are
   simulated by processing them as special operators. The double dot (..) is used to subtract a previous path 
   element, and the operator single dot (.) is for current dir. The dot operator is simply ignored to produce
   the correct results in directories. The file path /WDS1/file/. will result in the file /WDS1/file which isn't 
   entirely correct, but handling better isn't terribly productive and wastes precious resources.


   Examples:
   CWD                     REQ PATH               FINAL PATH
   ----------------------- ---------------------- ------------------------------
   - /DSK2                 GREP                   /DSK2/GREP
   - /                     DSK2.GREP              /DSK2/GREP
   - /WDS1                 ./X                    /WDS1/X

   Before these paths can be used for accessing volumes, directories and files on the TI-99/4A via DSRs, they
   must be transformed into the TI path format using the method generate_ti_path ().

   change log:
   08/14/2023 initial version
   08/16/2023 updated to handle dot (.) as cwd in the req_path
   11/13/2023 handle tilde for the home directory
   11/15/2023 added handling of root path
   11/16/2023 removed dead code
   11/21/2023 updated to use FILENAME_MAX
   12/10/2023 removed reference to string_ext.h
   02/24/2024 broad changes for cache and dylib
   06/23/2025 updated to handle all operations with internal variables until the result is proven valid, then copy to output.
              this was done to prevent overrunning buffers. also increased the number of path elements from 8 to 16.
   06/24/2025 added tests for no path element longer than 10 characters
   07/02/2025 relaxed length limitation on resultant filename
*/

#include <stdbool.h>
#include <string.h>
#include <stdio.h>
#include <unistd_private.h>
#include <cache_private.h>

void generate_unix_path (char *cwd_path, char *req_path, bool *final_path_ok, char *final_path) {

   char tfinal_path[64];

   char tcwd_path[FILENAME_MAX];
   char *pcwd_path[16];
   int ncwd_path = 0;
   char *p;

   // collect the cwd path elements
   if (*req_path != '/') {
      if (*req_path == '~') {
         strcpy (tcwd_path, cache.login.home_dir);
         req_path++;
      } else {
         strcpy (tcwd_path, cwd_path);
      }
      p = strtok (tcwd_path, "/");
      while (p) {
         pcwd_path[ncwd_path] = p;
         ncwd_path++;
         p = strtok (NULL, "/");
      }
   }

   char treq_path[FILENAME_MAX];
   char *preq_path[8];
   int nreq_path = 0;

   // collect the requested path operators and path elements
   strcpy (treq_path, req_path);
   p = strtok (treq_path, "/");
   while (p) {
      preq_path[nreq_path] = p;
      nreq_path++;
      p = strtok (NULL, "/");
   }

   // combine and process path elements and operators
   *final_path_ok = true;
   for (int i = 0; i < nreq_path; i++) {
      if (!strcmp (preq_path[i], ".")) {
      
      } else if (!strcmp (preq_path[i], "..")) {
         ncwd_path--;
         if (ncwd_path < 0) {
            *final_path_ok = false; // not ok to have a negative number of elements while processing
            break;
         }
      } else {
         pcwd_path[ncwd_path] = preq_path[i];
         ncwd_path++;
      }
   }

   // generate the final path
   if (*final_path_ok) {
      if (ncwd_path) {
        strcpy (tfinal_path, "");
        for (int i = 0; i < ncwd_path; i++) {
           if (strlen (pcwd_path[i]) > 10) {    /// ensure no path element is longer than 10 characters per the TI file spec
              *final_path_ok = false;
              break;
           }
           strcat (tfinal_path, "/");
           strcat (tfinal_path, pcwd_path[i]);
        }
      } else {
        strcpy (tfinal_path, "/"); // if there's no path elements then this path is root
      }
   }

   *final_path_ok = *final_path_ok && (strlen (tfinal_path) < FILENAME_MAX);

   if (*final_path_ok) {
      strcpy (final_path, tfinal_path);
   }
}
