/* getopt.c
   This method provides the getopt functionality for simple processing of command line arguments.
   change log:
   06/23/2023 initial version
   08/18/2023 added support for the suboption being contained as a continuation of the main option
   08/19/2023 updated to print the program name without full path
   08/20/2023 updated to use dylib.dylib.dylib.fputs/fputc rather than fprintf to reduce memory footprint
   12/10/2023 removed reference to string_ext.h
   02/27/2024 modified to use the dynamic library
   03/01/2024 modified to write to dylib.dylib.dylib.fputs solely with strings and no literals
   01/20/2025 updated dylib.fputs to dylib
   06/02/2025 added dylib reference
   06/04/2025 updated to support packed options such as -xyz, -xyza55, -xyza 55)
   06/07/2025 updated to fully use dylib
*/

#include <string.h>
#include <stdio.h>
#include <unistd.h>
#include <dylib.h>

// #define DEBUG

int optind       = 1;
char *optarg     = NULL;
char *optpack    = NULL;

char *get_program_name (const char *arg) {
   char *r    = NULL;
   char *name = (char *) arg;
   while ((name = strstr (name, "/"))) {
      name++;
      r = name;
   }
   return r;
}

const char *msg_illegal = ": illegal option -- ";
const char *msg_cr      = "\n";
const char *msg_optarg  = ": option requires an argument -- ";

int getopt (int argc, char * const argv[], const char *optstring) {
   char *p, *a;
   int r = 0;

   #ifdef DEBUG
      if (optpack) {
         fprintf (stdout, "old [%s]\n", optpack);
      }
   #endif

   // shift the packed options to the left
   if (optpack && dylib.strlen (optpack)) {
      optpack++;
      #ifdef DEBUG
         fprintf (stdout, "next [%s]\n", optpack);
      #endif
   }

   if (!optpack || !dylib.strlen (optpack)) {            
      if (optind < argc && argv[optind][0] == '-') {
         optpack = &argv[optind][1];
         optind++;
         #ifdef DEBUG
            fprintf (stdout, "new [%s]\n", optpack);
         #endif
      } else {
         r = -1;
      }
   }

   if (!r) {
      r = optpack[0];                  // yes, capture it

      p = dylib.strchr (optstring, r);      // assess whether its in the optstring (the list of valid options)
      if (p) {                 
         a = p;                       // it is, now look to see if there's a required suboption
         a++;
         if (*a == ':') {        
            // there is--it might be within the same argument (-csomething) or after (-c something)
            if (optpack[1] != 0x00) {  // test for being in the same argument
               optarg = &optpack[1];   // it is, capture it
            } else if (optind < argc) {
               optarg = argv[optind];   // it's in the next argument
               optind++;
            } else {                        // or wasn't provided at all
               dylib.fputs (argv[0], stderr);
               dylib.fputs (msg_optarg, stderr);
               fputc (r, stderr);
               dylib.fputs (msg_cr, stderr);
               r = '?';
               optarg = NULL;
            }
            optpack[0] = 0x00; 
         } else {
            optarg = NULL;
         }
      } else {
         dylib.fputs (argv[0], stderr);
         dylib.fputs (msg_illegal, stderr);
         fputc (r, stderr);
         dylib.fputs (msg_cr, stderr);
         r = '?';
      }
   } else {
      r = -1;
   }

   return r;
}
