/* groff.c
   This file implemented a simplistic subset of (G)ROFF capabilities.
   change log
   07/23/2025 initial version
   07/24/2025 corrected indentation calculation
              updated write_word to work as the tab for indentation
              updated to reinitialize first_word on beginning of line, after tabs, and after indents
   10/04/2025 corrected dylib refs
*/

#include <groff.h>
#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <stdbool.h>
#include <dylib.h>

#define GROFF_TAB_MAX 12

int col_width;
int col;
int tab[GROFF_TAB_MAX];
int tab_index;
bool first_word;

void groff_init (int height, int width) {
   const int tab_default[GROFF_TAB_MAX] = {0, 5, 10, 15, 20, 25, 30, 35, 40, 45, 50, 55};
   col_width  = width;
   col        = 0;
   first_word = true;
   tab_index    = 0;
   for (int i = 0; i < GROFF_TAB_MAX; i++) {
      tab[i] = tab_default[i];
   }
}

void groff_write_cr (FILE *fout) {
   dylib.fputs ("\n", fout);
   col        = 0;
   tab_index  = 0;
   first_word = true;
}

void groff_write_word (const char *s, FILE *fout) {
   int len = dylib.strlen (s);
   if (col + len + 1 >= col_width) {
      groff_write_cr (fout);
   }

   if (!col) {                         // tab over appropriately for this line
      while (col < tab[0]) {
         dylib.fputs (" ", fout);
         col++;
      }
      first_word = true;
   }

   if (!first_word) {
      dylib.fputs (" ", fout);
      col++;
   }
   first_word = false;

   dylib.fputs (s, fout);
   col += len;
}

void groff_write_tab (FILE *fout) {
   tab_index++;
   while (col < tab[tab_index]) {
      dylib.fputs (" ", fout);
      col++;
   }
   first_word = true;
}

void groff (FILE *fin, FILE *fout, int height, int width) {

   bool in_tp = false;

   char s[1024];
   char *t;
   char *p;

   bool in_word;
   
   bool first_paragraph = true;

   groff_init (height, width);

   while (dylib.fgets (s, sizeof (s), fin)) {

      t = dylib.strtok (s, " \n");
      if (t) {
         if (!strcmp (t, ".PP")) {
            if (!first_paragraph) {
               groff_write_cr (fout);
            }
            first_paragraph = false;
            in_tp = false;
            tab[0] = 0;                          // clear any indentation previously in effect
            tab_index = 0;                       // reset the tab index
            col = 0;
         } else if (!strcmp (t, ".TP")) {
            if (!first_paragraph) {
               groff_write_cr (fout);
            }
            first_paragraph = false;
            tab[0] = 0;                          // clear any indentation previously in effect
            tab_index = 0;                       // reset the tab index
            col = 0;
            in_tp = true;
         } else if (!strcmp (t, ".ta")) {
            tab_index = 1;
            while (1) {
               t = dylib.strtok (NULL, " \n");
               if (t) {
                  tab[tab_index] = atoi (t);
                  tab_index++;
               } else {
                  break;
               }
            }
            tab_index = 0;
         } else if (!strcmp (t, ".br")) {
            groff_write_cr (fout);
         } else {
            while (t) {
               p = t;
               in_word = false;
               while (*p) {
                  if (*p == '\t') {
                     if (in_word) {
                        *p = 0x00;
                        groff_write_word (t, fout);
                        in_word = false;
                     }
                     groff_write_tab (fout);
                     if (in_tp) {
                        tab[0] = tab[tab_index];
                     }
                     p++;
                     t = p;
                  } else {
                     in_word = true;
                     p++;
                  }
               }
               if (in_word) {
                  groff_write_word (t, fout);
                  in_word = false;
               }
      
               t = dylib.strtok (NULL, " \n");
            }
         }
      }
   }

   if (col) {
      groff_write_cr (fout);
   }
}
