/* image_generate.c
   This method generates a displayable image given a bitmap described in a text array.
   change log:
   11/16/2023 initial version
   03/01/2024 updated to use dylib
   07/05/2025 dylib adjustments
*/

#include <image.h>
#include <image_private.h>
#include <console.h>
#include <string.h>
#include <dylib.h>

int image_generate (image_t *image, const char *text_bitmap[]) {

   int r = 0;

   int text_bitmap_height, text_bitmap_width;

   int char_def_width = console_font_get_width ();

   text_bitmap_height = 0;
   for (int j = 0; j < IMAGE_CHAR_HEIGHT_MAX * IMAGE_CHAR_PIXEL_WIDTH_AND_HEIGHT; j++) {
      if (text_bitmap[j]) {
         text_bitmap_height = j + 1;
      } else {
         break;
      }
   }
   text_bitmap_width = dylib.strlen (text_bitmap[0]);

   if (text_bitmap_height <= IMAGE_CHAR_HEIGHT_MAX * IMAGE_CHAR_PIXEL_WIDTH_AND_HEIGHT &&
       text_bitmap_width <= IMAGE_CHAR_WIDTH_MAX * char_def_width) {

      unsigned char bitmap[IMAGE_CHAR_HEIGHT_MAX][IMAGE_CHAR_WIDTH_MAX][IMAGE_CHAR_DEF_SIZE];
      int pixel_row, pixel_col;
      int char_row, char_col, byte_row, bit_col;
      unsigned char bit;
      int i;

      memset (bitmap, 0x00, sizeof (bitmap));

      byte_row = 0;
      for (pixel_row = 0; pixel_row < text_bitmap_height; pixel_row++) {
         char_row = pixel_row / IMAGE_CHAR_PIXEL_WIDTH_AND_HEIGHT;
         bit_col = 0;
         for (pixel_col = 0; pixel_col < text_bitmap_width; pixel_col++) {
            char_col = pixel_col / char_def_width;
            if (text_bitmap[pixel_row][pixel_col] == 'X') {
               bit = 0x01;
            } else {
               bit = 0x00;
            }
            bitmap[char_row][char_col][byte_row] |= bit << (7 - bit_col);
            bit_col++;
            if (bit_col == char_def_width) {
               bit_col = 0;
            }
         }
         byte_row++;
         if (byte_row == IMAGE_CHAR_PIXEL_WIDTH_AND_HEIGHT) {
            byte_row = 0;
         }
      }

      memset (image, 0x00, sizeof (image_t));
      image->height = (text_bitmap_height + 7) / IMAGE_CHAR_PIXEL_WIDTH_AND_HEIGHT;
      image->width  = (text_bitmap_width + char_def_width - 1) / char_def_width;

      bool has_pixels_set;
      for (char_row = 0; char_row < image->height; char_row++) {
         for (char_col = 0; char_col < image->width; char_col++) {
            has_pixels_set = false;
            for (i = 0; i < IMAGE_CHAR_DEF_SIZE; i++) {
               has_pixels_set |= bitmap[char_row][char_col][i];
            }
            if (has_pixels_set) {
               image->id[char_row][char_col] = image_get_next_ch_id ();
               console_font_set (image->id[char_row][char_col], bitmap[char_row][char_col]);
            }
         }
      }
   } else {
      r = -1;
   }

   return r;
}
