/* malloc.c
   These methods provide heap-based dynamic memory management.
   change log:
   06/23/2023 initial version
   09/02/2023 moved definitions for storage to stdlib_private.h
   09/02/2023 moved free to separate file to save program space for programs not bothering to call it
   10/21/2023 modified to auto-initialize the malloc package on call
   10/21/2023 removed use of memory library
*/

#include <stdlib.h>
#include <stdio.h>
#include <stdlib_private.h>
#include <stdbool.h>

malloc_t *upper; // upper memory 
malloc_t *lower; // lower memory

// sets the allocation unit values (available/not) from the given index for count
void malloc_unit_set (malloc_t *m, int start_index, int count, bool value) {
   for (int i = start_index; i < start_index + count; i++) {
      m->avail[i] = value;
   }
}

// initializes all the malloc structures
void malloc_init_regions (unsigned int lower_start_addr, int lower_len, unsigned int upper_start_addr, int upper_len) {
   lower             = (malloc_t *)lower_start_addr;
   lower->count      = (lower_len - sizeof (malloc_t)) / (ALLOC_UNIT_SIZE + sizeof (bool));
   lower->start_addr = lower_start_addr + sizeof (malloc_t) + sizeof (bool) * lower->count;
   malloc_unit_set (lower, 0, lower->count, true);

   upper             = (malloc_t *)upper_start_addr;
   upper->count      = (upper_len - sizeof (malloc_t)) / (ALLOC_UNIT_SIZE + sizeof (bool));
   upper->start_addr = upper_start_addr + sizeof (malloc_t) + sizeof (bool) * upper->count;
   malloc_unit_set (upper, 0, upper->count, true);
}

void malloc_init () {
   static bool malloc_is_initialized = false;
   extern unsigned int _init_data;
   unsigned int *p;
   unsigned int text_start;
   unsigned int text_data_init;
   unsigned int text_end;
   unsigned int data_start;
   unsigned int data_end;
   unsigned int bss_start;
   unsigned int bss_end;
   unsigned int malloc_upper_start;
   unsigned int malloc_upper_len;
   unsigned int malloc_lower_start;
   unsigned int malloc_lower_len;

   // CPU RAM / ROM organization:
   // Start    End    Description  Notes
   // 0x2000 - 0x3fff Bottom 8 KB  data first, then bss are allocated here. Then stack at the top. A typical program may have
   //                              upwards of 2KB available for lower heep allocated after bss. There's always danger the
   //                              lower heap and stack will collide.
   // 0x6000 - 0x7fff ROM          Although not RAM this is included as the calculations for text_start/end are made in this
   //                              package. 
   // 0xa000 - 0xffff Top 24 KB    EA5 programs usually load here. Heap can be declared after the text and data init sections

   // The text_start is assertained based on the data initialization start address which part of the text section. If in the 
   // ROM range then text is 0x6000-XXXX, where XXXX <= 0x7fff; otherwise text is 0xa000-XXXX, where XXXX <= 0xffff.

#define STACK_BOTTOM_ADDR 0x3400

   if (!malloc_is_initialized) {

      // collect text, data and bss from _init_data defined in _start
      p = &_init_data;

      // p has start of data section
      data_start = *p;
      p++;
   
      // p has location of initial contents in text section. this is after the text section
      text_data_init = *p;
      p++;
   
      // p has size of data section
      text_end = text_data_init + *p - 1;
      data_end = data_start + *p - 1;
      p++;
   
      // address ea5 or rom
      if ((text_end >= 0xa000) && (text_end <= 0xffff)) {
         // this is an ea5 program
         text_start          = 0xa000;
         malloc_upper_start  = text_end + 1;
         // ensure the start is even
         if (malloc_upper_start % 2) {
            malloc_upper_start++;
         }
      } else {
         // this is a rom
         text_start         = 0x6000;
         malloc_upper_start = 0xa000;
      }
      malloc_upper_len   = 0x10000 - malloc_upper_start;
   
      // p has the start of bss
      bss_start = *p;
      p++;
   
      // p has the length of bss
      bss_end = bss_start + *p - 1;
   
      malloc_lower_start = bss_end + 1;                             // start of the lower region
      if (malloc_lower_start < STACK_BOTTOM_ADDR) {                 // make sure there's space available
         malloc_lower_len = STACK_BOTTOM_ADDR - malloc_lower_start; // save how much space
      } else {                                                      // there's no space to use
         malloc_lower_len = 0;                                      // so set
      }
   
      malloc_init_regions (malloc_lower_start, malloc_lower_len, malloc_upper_start, malloc_upper_len);

      malloc_is_initialized = true;
   }
}
   
// core malloc - locates available storage in a region
void *malloc_region (malloc_t *m, int count_reqd) {
   int count_found = 0;
   int rindex      = 0;
   void *r         = NULL;
   int i;
   alloc_info_t *a;

   // loop through all the region, locating a string of available units of the reqd count
   for (i = 0; i < m->count; i++) {
      if (m->avail[i]) {                   // found an available unit
         if (!count_found) {
            rindex = i;                    // this is the first found, so record the return index
         }
         count_found++;                    // add to the count found
         if (count_found == count_reqd) {
            break;                         // enough available units have been found so exit the loop
         }
      } else {
         count_found = 0;                  // reset the count since this unit isn't available
      }
   }

   if (count_found == count_reqd) {
      malloc_unit_set (m, rindex, count_reqd, false);          // set all the allocation units to unavailable
      r = (void *) (m->start_addr + rindex * ALLOC_UNIT_SIZE); // calculate the memory address to be returned
      a = (alloc_info_t *)r;                                   // recast of r to alloc_unit_t
      a->m           = m;                                      // retain the memory region
      a->start_index = rindex;                                 // retain the start index
      a->count       = count_reqd;                             // retain the count
      r += sizeof (alloc_info_t);                              // finally advance r beyond alloc_info 
   }
   return r;                                                   // return the location, or NULL if not
}

// externally available malloc - allocates memory from either upper or lower memory as available
void *malloc (int size) {

   malloc_init ();

   int count_reqd  = 
      (size + sizeof (alloc_info_t) + ALLOC_UNIT_SIZE - 1) /  // calculate the number of allocation units required.
      ALLOC_UNIT_SIZE;                                        // When doing so, add the size of one unit less one, then the 
                                                              // division will round the correct number
                                                              // size of the request, storage for alloc_info_t
   void *r = malloc_region (lower, count_reqd);                  // try to allocate in lower memory
   if (!r) {
      r = malloc_region (upper, count_reqd);               // try to allocate in upper memory
   }
   return r;                                                  // return the location, or NULL if none found
}
