/* more.c
   This method implements the more command, writing text one window-full at a time.
   change log
   07/01/2023 initial version
   07/10/2023 added error messages for files that cannot be opened
   07/26/2023 added output notifying that more is expecting user key press, and
              waiting for a final user key press when the file is complete.
              Also return if the main program should continue feeding additional files
              (when the user didn't press C).
   08/20/2023 updated to use dylib.fputs rather than fprintf
   12/10/2023 removed reference to string_ext.h
   02/29/2024 added use of dylib
              updated to use constant strings with dylib.fputs to prevent replacement of fwrite
   01/20/2025 updated fputs to dylib
   06/09/2025 corrected issue where short files would print a blank line before printing the file contents
   06/11/2025 major refactor, added print_over option
   07/23/2025 corrected bug handling blank lines
   10/04/2025 corrected ioctl.h ref
*/

#include <string.h>
#include <stdio.h>
#include <stdbool.h>
#include <console.h>
#include <sys/ioctl.h>
#include <dylib.h>

struct winsize w;                                              // window size
int y, x;                                                      // current print y, x

#define TAB_STOP 5                                             // tab stops
void expand_tabs (const char *in, char *out) {                 // expand tabs to spaces

   char *p = (char *) in;                                      // pointer p within in, initialized to the beginning of in
   char *q = out;                                              // pointer q within out, initialized to the beginning of out

   int x = 0;                                                  // position within out

   if (*p) {
      while (*p) {                                                // loop through all of in using p
         if (*p == 13) {                                          // if on a tab
            int xtab_count = TAB_STOP - (x % TAB_STOP);           // calculate the number of characters until the next tab stop
            while (xtab_count) {                                  // loop through the expanded tab to add spaces
               *q = ' ';                                          // set a space
               q++;                                               // increment the position
               *q = 0x00;                                         // null terminate
               x++;                                               // increment the position in out
               xtab_count--;                                      // decrement the expanded tab count
            }
         } else {                                                 // not a tab, just copy input
            *q = *p;                                              // copy p to q
            q++;                                                  // increment q
            *q = 0x00;                                            // null terminate out via q
         }
         p++;                                                     // increment p in in
      }
   } else {
      *q = 0x00;
   }
}

void wait_for_key (const char *filename) {                     // prompts the user that more is waiting for a key press

   dylib.fputs ("[more", stdout);                              // write the prompt with filename if provided
   if (filename) {
      dylib.fputs (" ", stdout);
      dylib.fputs (filename, stdout);
   }
   dylib.fputs ("]", stdout);

   console_getc ();                                            // wait for a key press                    

   dylib.fputs ("\r                        \r", stdout);       // remove the prompt
}

void write (const char *filename, char *s, bool print_over) {  // writes a complete line of text, waiting for user input after a screen full, and clearing if requested
   dylib.fputs (s, stdout);                                    // write the text -- this should be either a full line of text, a partial line of text with line feed, or
                                                               // just a line feed -- always it is a complete line of text
   y++;                                                        // increment the y line position
   if (y == w.ws_row - 1) {                                    // if at the bottom of the screen,
      wait_for_key (filename);                                 // wait for a user key press
      y = 0;                                                   // reset the y line position
      if (print_over) {                                        // if print over option is selected,
         dylib.fputs ("\f", stdout);                           // print a form feed, which will clear the screen
      }
   }
}

const char *more_prompt5 = "more: ";                           // find not found diagnostic messages
const char *more_prompt6 = ": No such file or directory\n";
const char *more_prompt7 = "\n";

void more (const char *filename, bool print_over) {            // the more procedure
   char incoming[256];                                         // string for input text
   char s[256];                                                // string for processing, after expanding tabs

   ioctl (fileno (stdout), TIOCGWINSZ, &w);                    // get the size of the screen window

   FILE *f;
   if (filename) {                                             // if a filename was supplied,
      f = dylib.fopen (filename, "r");                         // open the file
      if (!f) {                                                // if the file didn't open, output diagnostic information
         dylib.fputs (more_prompt5, stderr);
         dylib.fputs (filename, stderr);
         dylib.fputs (more_prompt6, stderr);
      }
   } else {
      f = stdin;                                               // no file supplied, so use stdin
   }

   if (f) {                                                    // if a file is open,
      y = 0;                                                   // reset the print position
      if (print_over) {                                        // if the print over option is selected,
         dylib.fputs ("\f", stdout);                           // print a form feed which will clear the screen
      }

      bool has_text_lines = false;                             // flag to indicate if text lines were processed

      while (dylib.fgets (incoming, sizeof (incoming), f)) {   // loop through all the lines of the file
         incoming[dylib.strcspn(incoming, "\r\n")] = 0x00;     // remove CR/LF

         expand_tabs (incoming, s);                            // expand tabs to spaces ahead of processing

         has_text_lines = true;                                // indicate text lines were processed

         char out[82];                                         // maximum screen width + 2
         char *p = s;                                          // pointer within s, initialized to the beginning of s
         char *q = out;                                        // pointer within out, initialized to the beginning of out

         *q = 0x00;                                            // null terminate out
         x = 0;                                                // reset the position to the left

         if (*p) {                                             // if p doesnt point to a null char, then
            while (*p) {                                       // process all of s, looping through s with p
               *q = *p;                                        // copy char
               q++;                                            // increment ptr in out
               *q = 0x00;                                      // null terminate out
               p++;                                            // increment ptr in s
               x++;                                            // increment screen column
               if (x == w.ws_col) {                            // if x has reached the right side of the screen (have a full line of text), then
                  write (filename, out, print_over);           // write the line of text
                  q = out;                                     // reset q back to the first position in out
                  x = 0;                                       // reset the screen x position
               }
            }

            if (x) {                                           // if x isn't at the left most position, then out has text not yet written to the screen:
               *q = '\n';                                      // add carriage return
               q++;                                            // increment the pointer
               *q = 0x00;                                      // and terminate out
               write (filename, out, print_over);              // write the line of text
            }

         } else {                                              // the input line of text was empty except for a line feed, so is zero length here:
            write (filename, "\n", print_over);                // write a line feed
         }
      }
 
      if (y || !has_text_lines) {                              // if lines where printed since the last user key press or if no lines were printed at all,
         wait_for_key (filename);                              // wait for a user key press
      } 
     
      if (filename) {                                          // if a filename was provided,
         dylib.fclose (f);                                     // close the file
      }
   }
}
