/* opendir.c
   This method opens a directory handle for reading directory contents.
   06/23/2023 initial version
   08/14/2023 updated for unix paths
   11/21/2023 updated to use FILENAME_MAX
   11/23/2023 updated to use separate dsr
   12/10/2023 removed reference to string_ext.h
   02/24/2024 broad changes for cache and dylib
              added spinner
   02/25/2024 modified to use cache.dir_in_use
   03/01/2024 more use of dylib
   01/10/2025 updated to use dylib.file_get_available
   01/22/2025 performance improvements
   05/30/2025 handled TIPI DSR error where the open succeeds when the record lengths don't match. Issue to be submitted to @jedimatt42.
   06/24/2025 renamed file_is_valid_path to file_has_valid_dsr_name
   07/05/2025 dylib adjustments
   10/13/2025 changed ref from file_has_valid_dsr_name to file_get_dsr_cru
*/

#include <assert.h>
#include <vdp.h>
#include <string.h>
#include <stdio.h>
#include <stdio_private.h>
#include <dirent.h>
#include <dirent_private.h>
#include <unistd.h>
#include <unistd_private.h>
#include <spinner_private.h>
#include <cache_private.h>
#include <dylib.h>
#include <stddef.h>

DIR *opendir (const char *path) {

   DIR *r = NULL;

   if (!cache.dir_in_use) {

      dylib.spinner_read ();

      char final_ti_path[FILENAME_MAX];
      dylib.generate_final_path (final_ti_path, (char *) path, true);

      if (file_get_dsr_cru (final_ti_path)) {       // verify the final path has a valid cru

         dir.df = dylib.file_get_available (true);

         if (dir.df) {

            dylib.spinner_write ();

            dylib.strcpy (dir.df->dsr->pab_filename, final_ti_path);
            dir.df->dsr->pab.NameLength   = dylib.strlen (dir.df->dsr->pab_filename);
            dir.df->dsr->pab.pName        = (unsigned char *)dir.df->dsr->pab_filename;
            dir.df->dsr->pab.OpCode       = DSR_OPEN;
            dir.df->dsr->pab.Status       = DSR_TYPE_INPUT | DSR_TYPE_INTERNAL | DSR_TYPE_RELATIVE;
            dir.df->dsr->pab.RecordNumber = 0;
            dir.df->dsr->pab.ScreenOffset = 0x00;
            dir.df->dsr->pab.RecordLength = 0;   // will either be 38 for legacy and 146 for extended attributes
            dir.df->dsr->pab.VDPBuffer    = dir.df->dsr->vdp_data_buffer_addr;

            int fr = dylib.dsrlnk (&dir.df->dsr->pab, dir.df->dsr->vdp_pab_buffer_addr);
            if (!fr) {

               // read the record length 
               vdpmemread (dir.df->dsr->vdp_pab_buffer_addr + OFFSETOF (struct PAB, RecordLength), &dir.df->dsr->pab.RecordLength, 1);

               // fail out if the record size doesn't match
               if (dir.df->dsr->pab.RecordLength != 38 && dir.df->dsr->pab.RecordLength != 146) {
                  dylib.spinner_write ();

                  // this is not a directory -- close the file
                  dir.df->dsr->pab.OpCode = DSR_CLOSE;
                  dylib.dsrlnk (&dir.df->dsr->pab, dir.df->dsr->vdp_pab_buffer_addr);

                  fr = 1;
               }
            }

            if (!fr) {
               dylib.spinner_write ();    
  
               dir.df->dsr->pab.OpCode = DSR_READ;
               dylib.dsrlnk (&dir.df->dsr->pab, dir.df->dsr->vdp_pab_buffer_addr);
      
               char buf[150];
               
               // get the length of the read data
               vdpmemread (dir.df->dsr->vdp_pab_buffer_addr + 5, (unsigned char *) &dir.df->dsr->pab.CharCount, 1);                          
               // copy read data from vdp to ram
               vdpmemread (dir.df->dsr->vdp_data_buffer_addr, (unsigned char *) buf, dir.df->dsr->pab.CharCount);                   

               char *cursor = buf;
               char len;
      
               cursor = memcpy_cursor (&len, cursor, sizeof (char));
               cursor = memcpy_cursor (dir.volume, cursor, len);
               dir.volume[(int) len] = 0x00;
      
               double nop;
               cursor = memcpy_cursor (&len, cursor, sizeof (char));
               cursor = memcpy_cursor ((char *) &nop, cursor, sizeof (double));
      
               cursor = memcpy_cursor (&len, cursor, sizeof (char));
               cursor = memcpy_cursor ((char *) &dir.total_sectors, cursor, sizeof (double));
      
               cursor = memcpy_cursor (&len, cursor, sizeof (char));
               cursor = memcpy_cursor ((char *) &dir.free_sectors, cursor, sizeof (double));
      
               dir.df->dsr->pab.RecordNumber++;

               dir.df->dsr->in_use = true;
               dir.df->in_use      = true;
               cache.dir_in_use    = true;
                 
               r = &dir;
            } else {
               dir.df->in_use = false;
            }
         }
      }
      dylib.spinner_restore ();
   }

   return r;
}
